--[[
Improves the exhaust smoke effect by making it darker and more realistic. Mod inspired by darker exhaust smoke mod by Rival.

Author:     w33zl
Version:    1.2.0
Modified:   2026-01-16

Changelog:

]]

local FEATURETOGGLE_LOWTEMP = true
local FEATURETOGGLE_MOTORSTART = true

local DEBUG_MODE = false
-- DEBUG_MODE = g_modDebugMode or DEBUG_MODE

local ENGINE_LOAD_THRESHOLD_LOW = 0.4
local ENGINE_LOAD_THRESHOLD_HIGH = 0.95
local MAX_ENGINELOAD_BOOST = 0.8




local function RGB(r, g, b)
	return {r / 255, g / 255, b / 255 }
end

local COLOR_PROFILE_NODEF = {
	DEFAULT = {
		minColor = RGB(17, 17, 0),
		maxColor = RGB(10, 8, 0),
		minFactor = 1.8,
		maxFactor = 4.0,
	},
	LOW_TEMP = {
		minColor = RGB(5, 5, 2),
		maxColor = RGB(0, 0, 0),
		minFactor =  1.8,
		maxFactor = 4.0,
	},
}

local COLOR_PROFILE_MODERN = {
	DEFAULT = {
		minColor = RGB(220, 220, 220),
		maxColor = RGB(190, 190, 190),
		minFactor = 0.2,
		maxFactor = 0.5,
	},
	LOW_TEMP = { 
		minColor = RGB(190, 190, 168),
		maxColor = RGB(100, 100, 88),
		minFactor = 0.5,
		maxFactor = 1.0,
	},
}



--[[

Reference colors:
Fiat: minRpmColor="0 0 0 4" maxRpmColor="0 0 0 8"
Ford: minRpmColor="0.9 0.9 0.9 3" maxRpmColor="0.01 0.01 0.01 10" 
Deere: minRpmColor="0.9 0.9 0.9 0.1" maxRpmColor="0.9 0.9 0.9 0.4"
K700: minRpmColor="0.2 0.2 0.2 0.03" maxRpmColor="0.0 0.0 0.0 5"
Basegame: minRpmColor="0.9 0.9 0.9 0.1" maxRpmColor="0.9 0.9 0.9 0.4"

]]

ImprovedExhaustEffects = {}

-- function that enables improved exhaust effects on a exhaust effect
function ImprovedExhaustEffects:EnableImprovedExhaustEffects(exhaustEffect, isModern)
	if exhaustEffect.enableImprovedExhaustEffects then
		Log:debug("ImprovedExhaustEffects already enabled for this exhaust effect, skipping")
		return
	end

	if isModern then
		exhaustEffect.colorProfile = COLOR_PROFILE_MODERN
	else
		exhaustEffect.colorProfile = COLOR_PROFILE_NODEF
	end

	ImprovedExhaustEffects:ActivateDefaultColors(exhaustEffect)

	exhaustEffect.enableImprovedExhaustEffects = true
	
end

function ImprovedExhaustEffects:ActivateColorProfile(exhaustEffect, colorProfile)
	exhaustEffect.minRpmColor = {
		colorProfile.minColor[1],
		colorProfile.minColor[2],
		colorProfile.minColor[3],
		colorProfile.minFactor,
	}
	exhaustEffect.maxRpmColor = {
		colorProfile.maxColor[1],
		colorProfile.maxColor[2],
		colorProfile.maxColor[3],
		colorProfile.maxFactor,
	}
end

function ImprovedExhaustEffects:ActivateDefaultColors(exhaustEffect)
	if exhaustEffect.colorProfile == nil then
		Log:debug("ERROR: No color profile defined for exhaust effect")
		return
	end

	ImprovedExhaustEffects:ActivateColorProfile(exhaustEffect, exhaustEffect.colorProfile.DEFAULT)
end

function ImprovedExhaustEffects:ActivateLowTempColors(exhaustEffect)
	if exhaustEffect.colorProfile == nil then
		Log:debug("ERROR: No color profile defined for exhaust effect")
		return
	end

	ImprovedExhaustEffects:ActivateColorProfile(exhaustEffect, exhaustEffect.colorProfile.LOW_TEMP)
end

function ImprovedExhaustEffects:BoostFactor(exhaustEffect, factor)
	if exhaustEffect.minRpmColor ~= nil then
		exhaustEffect.minRpmColor[4] = exhaustEffect.minRpmColor[4] * factor
	end
	if exhaustEffect.maxRpmColor ~= nil then
		exhaustEffect.maxRpmColor[4] = exhaustEffect.maxRpmColor[4] * factor
	end
end



Motorized.onExhaustEffectI3DLoaded = Utils.appendedFunction(Motorized.onExhaustEffectI3DLoaded, function(self, ...)
	local isModern = false
	local consumesFuel = false
	local spec = self.spec_motorized
	local consumers = spec.consumersByFillTypeName

	if consumers ~= nil then
		for _, consumerType in pairs(consumers) do
			if consumerType.fillType == FillType.DEF or consumerType.fillType == FillType.METHANE then
				isModern = true
				-- break
			end
			if consumerType.fillType == FillType.DIESEL or consumerType.fillType == FillType.METHANE then
				consumesFuel = true
				-- break
			end
		end
	end

	if spec.exhaustEffects == nil or #spec.exhaustEffects == 0 then
		Log:debug("No exhaust effects found")
		return
	end

	if not consumesFuel then
		Log:debug("Vehicle doesn't consume fuel, no need to look at exhaust effects")
		return
	end


	for _, childExhaustEffect in ipairs(spec.exhaustEffects) do

		ImprovedExhaustEffects:EnableImprovedExhaustEffects(childExhaustEffect, isModern)

	end

	-- Log:table("spec.exhaustEffects", spec.exhaustEffects)
end)


if FEATURETOGGLE_LOWTEMP then
	Motorized.updateMotorTemperature = Utils.appendedFunction(Motorized.updateMotorTemperature, function(self, ...)

		--TODO: is this updated also when motor is off? if so, we could improves performance by skipping when motor is off

		local spec = self.spec_motorized

		if #spec.exhaustEffects == 0 then
			Log:debug("WARNING: No exhaust effect found for motor temperature update")
			return
		end

		for _, exhaustEffect in ipairs(spec.exhaustEffects) do

			local baseFactor = 1.0

			if spec.motorTemperature.value < spec.motorFan.disableTemperature then

				ImprovedExhaustEffects:ActivateLowTempColors(exhaustEffect)


				local minTemp = 0 --spec.motorTemperature.valueMin --NOTE: any benefit of using minValue?
				-- local maxTemp = spec.motorTemperature.valueMax -- TODO: We do not need this?
				local currentMotorTemp = math.max(spec.motorTemperature.value, minTemp)

				-- calculate a normalized factor based on motor temperature using minTemp and spec.motorFan.disableTemperature values
				local temp = spec.motorTemperature.value

				local motorTempFactor = 1 - (currentMotorTemp - minTemp) / (spec.motorFan.disableTemperature - minTemp)

				-- Calculate a clamped boost factor based on motor temperature using the range between TEMP_BOOST_LOW and TEMP_BOOST_HIGH
				local TEMP_BOOST_LOW = 1.0
				local TEMP_BOOST_HIGH = 2.4
				local tempBoostFactor = motorTempFactor * (TEMP_BOOST_HIGH - TEMP_BOOST_LOW) + TEMP_BOOST_LOW

				baseFactor = baseFactor * tempBoostFactor
		
				if FEATURETOGGLE_MOTORSTART then --NOTE: temporary disabled due to issues
					-- Boost the effect when starting the engine
					if self:getMotorState() == MotorState.STARTING then

						local motorStartFactor = math.random(50, 250) / 100

						baseFactor = baseFactor * motorStartFactor
					end
				end				

				
			else
				ImprovedExhaustEffects:ActivateDefaultColors(exhaustEffect)
			end


			-- Boost factor based on engine load
			
			-- Calculate a boost factor based on actual load percentage (min ENGINE_LOAD_THRESHOLD_LOW to max ENGINE_LOAD_THRESHOLD_HIGH), boost ranges from 1.0 to 1.2
			local loadPercentageClamped = math.min(math.max(spec.actualLoadPercentage, ENGINE_LOAD_THRESHOLD_LOW), ENGINE_LOAD_THRESHOLD_HIGH)
			local motorLoadFactor = 1.0 + ((loadPercentageClamped - ENGINE_LOAD_THRESHOLD_LOW) / (ENGINE_LOAD_THRESHOLD_HIGH - ENGINE_LOAD_THRESHOLD_LOW)) * MAX_ENGINELOAD_BOOST

			baseFactor = baseFactor * motorLoadFactor

			-- Update factor on min and max color
			ImprovedExhaustEffects:BoostFactor(exhaustEffect, baseFactor)

			Log:debug("temp: %f load: %f loadFactor: %f factor: %f", spec.motorTemperature.value, spec.actualLoadPercentage, motorLoadFactor, baseFactor)
		end
	end)
end

