--[[
Copyright (C) GtX (Andy), 2020

Author: GtX | Andy
Date: 05.05.2020
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

MultiplayerVehicleKeysManager = {}

MultiplayerVehicleKeysManager.MAXIMUM_KEYS = 60
MultiplayerVehicleKeysManager.KEYS_NUM_BITS = 6

local validationFail
local popupMessage

local buildId = 1
local versionString = "0.0.0.0"

local modName = g_currentModName
local modDirectory = g_currentModDirectory

local MultiplayerVehicleKeysManager_mt = Class(MultiplayerVehicleKeysManager)

function MultiplayerVehicleKeysManager.new()
    local self = setmetatable({}, MultiplayerVehicleKeysManager_mt)

    self.isServer = g_server ~= nil
    self.isClient = g_client ~= nil

    self.modName = modName

    self.buildId = buildId
    self.versionString = versionString

    self.numInserted = 0
    self.currentVehicleKeys = {}

    self.keyLimit = 2
    self.masterKeyEnabled = true

    self.gameStarted = false

    return self
end

function MultiplayerVehicleKeysManager:load()
    g_messageCenter:subscribe(MessageType.CURRENT_MISSION_START, self.onMissionStarted, self)

    g_messageCenter:subscribe(MessageType.PLAYER_NICKNAME_CHANGED, self.onPlayerNicknameChanged, self)
    g_messageCenter:subscribe(MessageType.PLAYER_FARM_CHANGED, self.onPlayerFarmChanged, self)

    if self.isServer then
        g_messageCenter:subscribe(MessageType.USER_REMOVED, self.onUserRemoved, self)
    end

    return true
end

function MultiplayerVehicleKeysManager:delete()
    self.gameStarted = false

    g_messageCenter:unsubscribeAll(self)

    if g_currentMission ~= nil then
        g_currentMission:removeUpdateable(self)
    end
end

function MultiplayerVehicleKeysManager:update(dt)
    if self.delayedKeyLimit ~= nil and self.gameStarted then
        self.updateTime -= dt

        if self.updateTime <= 0 then
            self:setKeyLimit(self.delayedKeyLimit)
            self.delayedKeyLimit = nil
        end
    else
        g_currentMission:removeUpdateable(self)
    end
end

function MultiplayerVehicleKeysManager:loadFromXMLFile(xmlFile, key)
    self:setKeyLimit(xmlFile:getInt(key .. ".settings#keyLimit", self.keyLimit), true, true)
    self:setMasterKeyEnabled(xmlFile:getBool(key .. ".settings#masterKey", self.masterKeyEnabled), true, true)
end

function MultiplayerVehicleKeysManager:saveToXMLFile(xmlFile, key)
    xmlFile:setString(string.format("%s#version", key), self.versionString)
    xmlFile:setFloat(string.format("%s#buildId", key), self.buildId)

    xmlFile:setInt(string.format("%s.settings#keyLimit", key), self.keyLimit)
    xmlFile:setBool(string.format("%s.settings#masterKey", key), self.masterKeyEnabled)
end

function MultiplayerVehicleKeysManager:setDelayedKeyLimit(keyLimit)
    if self.gameStarted then
        self.delayedKeyLimit = math.clamp(keyLimit or self.keyLimit, 1, MultiplayerVehicleKeysManager.MAXIMUM_KEYS)
        self.updateTime = 1000

        g_currentMission:addUpdateable(self)
    end
end

function MultiplayerVehicleKeysManager:setKeyLimit(keyLimit, noEventSend, suppressInfo)
    keyLimit = math.clamp(keyLimit or self.keyLimit, 1, MultiplayerVehicleKeysManager.MAXIMUM_KEYS)

    if self.keyLimit ~= keyLimit then
        self.keyLimit = keyLimit

        MultiplayerVehicleKeysSettingsEvent.sendEvent(false, keyLimit, self.masterKeyEnabled, nil, noEventSend)

        if not suppressInfo and self.gameStarted then
            Logging.info("MultiplayerVehicleKeys Setting 'keyLimit': %d", keyLimit)
        end
    end
end

function MultiplayerVehicleKeysManager:setMasterKeyEnabled(enabled, noEventSend, suppressInfo)
    enabled = Utils.getNoNil(enabled, self.masterKeyEnabled)

    if self.masterKeyEnabled ~= enabled then
        self.masterKeyEnabled = enabled

        MultiplayerVehicleKeysSettingsEvent.sendEvent(false, self.keyLimit, enabled, nil, noEventSend)

        if not suppressInfo and self.gameStarted then
            Logging.info("MultiplayerVehicleKeys Setting 'masterKey': %s", enabled)
        end
    end
end

function MultiplayerVehicleKeysManager:restoreAllVehicleKeys(targetVehicle, noEventSend)
    MultiplayerVehicleKeysSettingsEvent.sendEvent(true, nil, nil, targetVehicle, noEventSend)

    if targetVehicle == nil then
        local numReturned = 0

        for _, vehicle in ipairs(g_currentMission.vehicleSystem.enterables) do
            if vehicle.getHasKeyOwner ~= nil and vehicle:getHasKeyOwner() then
                vehicle:removeKeyOwner(true, true)

                numReturned = numReturned + 1
            end
        end

        if numReturned > 0 then
            Logging.info("[MultiplayerVehicleKeys] %d vehicle keys have been returned by an admin!", numReturned)
        end
    elseif targetVehicle.getHasKeyOwner ~= nil and targetVehicle:getHasKeyOwner() then
        targetVehicle:removeKeyOwner(true, true)

        Logging.info("[MultiplayerVehicleKeys] Vehicle (%s) keys have been returned by an admin!", targetVehicle:getFullName())
    end
end

function MultiplayerVehicleKeysManager:onSettingsChanged()
    if g_gui ~= nil and g_gui.currentGuiName == "InGameMenu" and g_inGameMenu ~= nil then
        local pageSettings = g_inGameMenu.pageSettings

        if pageSettings ~= nil and not pageSettings.isOpening then
            if pageSettings.multiplayerVehicleKeys_multiMaxKeys ~= nil then
                pageSettings.multiplayerVehicleKeys_multiMaxKeys:setState(self.keyLimit)
            end

            if pageSettings.multiplayerVehicleKeys_binaryMasterKey ~= nil then
                pageSettings.multiplayerVehicleKeys_binaryMasterKey:setIsChecked(self.masterKeyEnabled, false)
            end
        end
    end
end

function MultiplayerVehicleKeysManager:addKey(vehicle)
    self.currentVehicleKeys[vehicle] = vehicle
end

function MultiplayerVehicleKeysManager:removeKey(vehicle)
    self.currentVehicleKeys[vehicle] = nil
end

function MultiplayerVehicleKeysManager:onMissionStarted(isNewSavegame)
    local text = g_i18n:getText("info_holdingKeys", modName)
    local hud = g_currentMission.hud

    for _, vehicle in ipairs(g_currentMission.vehicleSystem.enterables) do
        if vehicle.getIsKeyOwner ~= nil and vehicle:getIsKeyOwner() then
            hud:addSideNotification(FSBaseMission.INGAME_NOTIFICATION_OK, string.namedFormat(text, "vehicleName", tostring(vehicle:getFullName())), 6000)
        end
    end

    Logging.info("MultiplayerVehicleKeys Setting 'keyLimit': %d", self.keyLimit)
    Logging.info("MultiplayerVehicleKeys Setting 'masterKey': %s", self.masterKeyEnabled)

    self.gameStarted = true
end

function MultiplayerVehicleKeysManager:onClientJoined(connection)
    if self.isServer and connection ~= nil then
        connection:sendEvent(MultiplayerVehicleKeysSettingsEvent.new(false, self.keyLimit, self.masterKeyEnabled, nil))
    end
end

function MultiplayerVehicleKeysManager:onPlayerNicknameChanged(player)
    if player == nil or g_currentMission == nil then
        return
    end

    local user = g_currentMission.userManager:getUserByUserId(player.userId)

    if user ~= nil then
        local nickname = user:getNickname()
        local uniqueUserId = user:getUniqueUserId()

        for _, vehicle in ipairs(g_currentMission.vehicleSystem.enterables) do
            local spec = vehicle.spec_enterableMultiplayerKeys

            if spec ~= nil and spec.keyOwnerUniqueUserId == uniqueUserId then
                spec.keyOwnerLastNickname = nickname
            end
        end
    end
end

function MultiplayerVehicleKeysManager:onPlayerFarmChanged(player)
    if player == g_localPlayer then
        local numReturned = 0

        for _, vehicle in ipairs(g_currentMission.vehicleSystem.enterables) do
            if vehicle.getIsKeyOwner ~= nil and vehicle:getIsKeyOwner() then
                vehicle:removeKeyOwner(true)
            end
        end

        if numReturned > 0 and self.gameStarted then
            Logging.info("[MultiplayerVehicleKeys] Farm changed, %d vehicle keys have been returned.", numReturned)
        end
    end
end

function MultiplayerVehicleKeysManager:onUserRemoved(user)
    if self.isServer and g_currentMission ~= nil and user:getIsBlocked() then
        local uniqueUserId = user:getUniqueUserId()

        if uniqueUserId ~= nil then
            for _, vehicle in ipairs(g_currentMission.vehicleSystem.enterables) do
                if vehicle.getIsKeyOwner ~= nil and vehicle:getIsKeyOwner(uniqueUserId) then
                    vehicle:removeKeyOwner(true)
                end
            end
        end
    end
end

function MultiplayerVehicleKeysManager:getCurrentKeys()
    return self.currentVehicleKeys
end

function MultiplayerVehicleKeysManager:getKeyLimit()
    return self.keyLimit
end

function MultiplayerVehicleKeysManager:getMasterKeyEnabled()
    return self.masterKeyEnabled
end

local function isActive()
    return g_modIsLoaded[modName] and g_multiplayerVehicleKeys ~= nil
end

local function getIsMultiplayerGame()
    if buildId > 0 then
        if g_dedicatedServerInfo ~= nil then
            return true
        end

        if g_careerScreen ~= nil then
            return g_careerScreen.isMultiplayer
        end
    end

    return true
end

local function validateMod()
    local mod = g_modManager:getModByName(modName)

    if mod == nil or g_iconGenerator ~= nil or g_isEditor then
        return false
    end

    versionString = mod.version or versionString

    if mod.modName == "FS25_MultiplayerVehicleKeys" or mod.modName == "FS25_MultiplayerVehicleKeys_update" then
        if mod.author ~= nil and #mod.author == 3 then
            return true
        end
    end

    validationFail = {
        startUpdateTime = 2000,

        update = function(self, dt)
            self.startUpdateTime = self.startUpdateTime - dt

            if self.startUpdateTime < 0 then
                local text = string.namedFormat(g_i18n:getText("ui_loadError", modName), "modName", mod.modName, "author", mod.author or "Unknown")

                if g_dedicatedServer == nil then
                    if not g_gui:getIsGuiVisible() then
                        local title = string.format("%s - Version %s", mod.title, versionString)
                        local yesText = g_i18n:getText("button_modHubDownload")
                        local noText = g_i18n:getText("button_ok")

                        YesNoDialog.show(self.openModHubLink, nil, text, title, yesText, noText, DialogElement.TYPE_LOADING)
                    end
                else
                    print("\n" .. text .. "\n    - https://farming-simulator.com/mods.php?&title=fs2025&filter=org&org_id=129652&page=0" .. "\n")
                    self.openModHubLink(false)
                end
            end
        end,

        openModHubLink = function(yes)
            if yes then
                openWebFile("mods.php?title=fs2025&filter=org&org_id=129652&page=0", "")
            end

            removeModEventListener(validationFail)
            validationFail = nil
        end
    }

    addModEventListener(validationFail)

    return false
end

local function validateTypes(typeManager)
    if typeManager.typeName == "vehicle" and isActive() then
        local numInserted = 0
        local specializationName = MultiplayerVehicleKeys.SPEC_NAME
        local specializationObject = g_specializationManager:getSpecializationObjectByName(specializationName)

        if specializationObject ~= nil then
            for typeName, typeEntry in pairs (typeManager:getTypes()) do
                if specializationObject.prerequisitesPresent(typeEntry.specializations) then
                    typeManager:addSpecialization(typeName, specializationName)

                    numInserted += 1
                end
            end
        end

        g_multiplayerVehicleKeys.numInserted = numInserted

        if numInserted > 0 then
            g_gui:loadProfiles(modDirectory .. "gui/guiProfiles.xml")

            InGameMenuSettingsFrame.inputEvent = Utils.overwrittenFunction(InGameMenuSettingsFrame.inputEvent, function(frame, superFunc, action, value, eventUsed)
                if action == InputAction.MENU_ACCEPT and frame.multiplayerVehicleKeys_buttonResetKeys ~= nil and FocusManager:getFocusedElement() == frame.multiplayerVehicleKeys_buttonResetKeys then
                    return false
                end

                return superFunc(frame, action, value, eventUsed)
            end)

            InGameMenuSettingsFrame.updateGameSettings = Utils.appendedFunction(InGameMenuSettingsFrame.updateGameSettings, function(frame)
                if not isActive() then
                    return
                end

                if frame.multiplayerVehicleKeys_sectionHeader == nil then
                    local gameSettingsLayout = frame.gameSettingsLayout

                    for i, element in ipairs (gameSettingsLayout.elements) do
                        if element:isa(TextElement) then
                            local sectionHeaderText = "Multiplayer Vehicle Keys"
                            local mod = g_modManager:getModByName(modName)

                            if mod ~= nil and (mod.title ~= nil and mod.title ~= "") then
                                sectionHeaderText = mod.title
                            end

                            frame.multiplayerVehicleKeys_sectionHeader = element:clone(gameSettingsLayout)
                            frame.multiplayerVehicleKeys_sectionHeader:setText(sectionHeaderText)

                            break
                        end
                    end

                    local multiTextOptionParent = nil
                    local binaryOptionParent = nil

                    for i, element in ipairs (gameSettingsLayout.elements) do
                        if #element.elements > 0 and element:isa(BitmapElement) then
                            if element.elements[1]:isa(BinaryOptionElement) then
                                if binaryOptionParent == nil then
                                    binaryOptionParent = element
                                end
                            elseif element.elements[1]:isa(MultiTextOptionElement) then
                                if multiTextOptionParent == nil then
                                    multiTextOptionParent = element
                                end
                            end

                            if binaryOptionParent ~= nil and multiTextOptionParent ~= nil then
                                break
                            end
                        end
                    end

                    if multiTextOptionParent ~= nil then
                        local texts = table.create(MultiplayerVehicleKeysManager.MAXIMUM_KEYS)

                        for i = 1, MultiplayerVehicleKeysManager.MAXIMUM_KEYS do
                            table.insert(texts, tostring(i))
                        end

                        local multiParent = multiTextOptionParent:clone(gameSettingsLayout, false)
                        local multiMaxKeys = multiParent.elements[1]

                        multiMaxKeys:setTexts(texts)

                        function multiMaxKeys.onClickCallback(_, keyLimit)
                            if g_multiplayerVehicleKeys ~= nil then
                                g_multiplayerVehicleKeys:setDelayedKeyLimit(keyLimit)
                            end
                        end

                        multiMaxKeys.id = "multiplayerVehicleKeys_multiMaxKeys"

                        multiMaxKeys:setDisabled(false)
                        multiMaxKeys:setVisible(true)

                        multiParent:setDisabled(false)
                        multiParent:setVisible(true)

                        multiParent.elements[2]:setText(g_i18n:getText("setting_maxVehicleKeys", modName))
                        multiMaxKeys.elements[1]:setText(g_i18n:getText("toolTip_maxVehicleKeys", modName))

                        frame.multiplayerVehicleKeys_multiMaxKeys = multiMaxKeys

                        if binaryOptionParent ~= nil then
                            local binaryParent = binaryOptionParent:clone(gameSettingsLayout, false)
                            local binaryMasterKey = binaryParent.elements[1]

                            function binaryMasterKey.onClickCallback(_, state)
                                if g_multiplayerVehicleKeys ~= nil then
                                    g_multiplayerVehicleKeys:setMasterKeyEnabled(state == BinaryOptionElement.STATE_RIGHT)
                                end
                            end

                            binaryMasterKey.id = "multiplayerVehicleKeys_binaryMasterKey"
                            binaryMasterKey:setDisabled(false)
                            binaryMasterKey:setVisible(true)

                            binaryParent:setDisabled(false)
                            binaryParent:setVisible(true)

                            binaryMasterKey:setIsChecked(true, true)
                            binaryMasterKey:updateSelection()

                            binaryParent.elements[2]:setText(g_i18n:getText("setting_masterUserKey", modName))
                            binaryMasterKey.elements[1]:setText(g_i18n:getText("toolTip_masterUserKey", modName))

                            frame.multiplayerVehicleKeys_binaryMasterKey = binaryMasterKey
                        end

                        if frame.buttonPauseGame ~= nil then
                            local buttonParent = frame.buttonPauseGame.parent:clone(gameSettingsLayout, false)

                            local buttonElement = buttonParent.elements[1]
                            local buttonTitleElement = buttonParent.elements[2]

                            if buttonElement ~= nil and buttonTitleElement ~= nil then
                                local buttonToolTipElement = multiMaxKeys.elements[1]:clone(buttonElement, false)

                                buttonToolTipElement:setText(g_i18n:getText("toolTip_resetVehicleKeys", modName))
                                buttonTitleElement:setText(g_i18n:getText("setting_resetVehicleKeys", modName))
                                buttonElement:setText(g_i18n:getText("button_reset"))

                                buttonElement.id = "multiplayerVehicleKeys_buttonResetKeys"
                                buttonElement.name = "buttonResetKeys"

                                function buttonElement.onClickCallback()
                                    if g_multiplayerVehicleKeys ~= nil then
                                        local yesText = g_i18n:getText("button_confirm")
                                        local noText = g_i18n:getText("button_cancel")

                                        local currentVehicle = g_localPlayer:getCurrentVehicle()

                                        local function restoreKeysCallback(yes)
                                            if yes then
                                                g_multiplayerVehicleKeys:restoreAllVehicleKeys(currentVehicle)

                                                if currentVehicle == nil then
                                                    InfoDialog.show(g_i18n:getText("info_returnedAllKeys", modName))
                                                else
                                                    InfoDialog.show(string.namedFormat(g_i18n:getText("info_returnedEnteredKeys", modName), "vehicleName", tostring(currentVehicle:getFullName())))
                                                end
                                            end
                                        end

                                        if currentVehicle == nil then
                                            YesNoDialog.show(restoreKeysCallback, nil, g_i18n:getText("warning_returnAllKeys", modName), nil, yesText, noText)
                                        else
                                            YesNoDialog.show(restoreKeysCallback, nil, string.namedFormat(g_i18n:getText("warning_returnEnteredKeys", modName), "vehicleName", tostring(currentVehicle:getFullName())), nil, yesText, noText)
                                        end
                                    end
                                end

                                buttonElement.isAlwaysFocusedOnOpen = false
                                buttonElement:reset()

                                GuiOverlay.deleteOverlay(buttonElement.icon)
                                buttonElement.icon = {}

                                buttonElement:applyProfile("multiplayerVehicleKeys_resetKeysButton")

                                buttonElement:setDisabled(false)
                                buttonElement:setVisible(true)

                                buttonParent:setDisabled(false)
                                buttonParent:setVisible(true)

                                frame.multiplayerVehicleKeys_buttonResetKeys = buttonElement
                            end
                        end
                    end

                    gameSettingsLayout:invalidateLayout()
                end

                if frame.multiplayerVehicleKeys_multiMaxKeys ~= nil then
                    frame.multiplayerVehicleKeys_multiMaxKeys:setState(g_multiplayerVehicleKeys:getKeyLimit())
                end

                if frame.multiplayerVehicleKeys_binaryMasterKey ~= nil then
                    frame.multiplayerVehicleKeys_binaryMasterKey:setIsChecked(g_multiplayerVehicleKeys:getMasterKeyEnabled(), true)
                end
            end)
        end
    end
end

local function loadMapFinished(currentMission)
    if isActive() and g_multiplayerVehicleKeys.numInserted > 0 then
        if g_multiplayerVehicleKeys:load() then
            local missionInfo = currentMission.missionInfo

            if missionInfo ~= nil and missionInfo.savegameDirectory ~= nil then
                local xmlFilename = missionInfo.savegameDirectory .. "/multiplayerVehicleKeys.xml"
                local xmlFile = XMLFile.loadIfExists("multiplayerVehicleKeysXML", xmlFilename)

                if xmlFile ~= nil then
                    g_multiplayerVehicleKeys:loadFromXMLFile(xmlFile, "multiplayerVehicleKeys")

                    xmlFile:delete()
                end
            end
        end
    end
end

local function unloadVehicles(vehicleSystem)
    if isActive() then
        g_multiplayerVehicleKeys:delete()

        if g_globalMods ~= nil then
            g_globalMods.multiplayerVehicleKeys = nil
        end

        g_multiplayerVehicleKeys = nil
    end
end

local function saveItems()
    if isActive() then
        local xmlFilename = g_currentMission.missionInfo.savegameDirectory .. "/multiplayerVehicleKeys.xml"
        local xmlFile = XMLFile.create("multiplayerVehicleKeysXML", xmlFilename, "multiplayerVehicleKeys")

        if xmlFile ~= nil then
            g_multiplayerVehicleKeys:saveToXMLFile(xmlFile, "multiplayerVehicleKeys")

            xmlFile:save()
            xmlFile:delete()
        end
    end
end

local function sendInitialClientState(_, connection, user, farm)
    if isActive() then
        g_multiplayerVehicleKeys:onClientJoined(connection)
    end
end

local function init()
    if g_globalMods == nil then
        g_globalMods = {}
    end

    if g_globalMods.multiplayerVehicleKeys then
        Logging.error("Failed to load mod '%s', 'MultiplayerVehicleKeys' specialisation was already loaded by '%s'!", modName, g_globalMods.multiplayerVehicleKeys.modName or "N/A")

        return
    end

    if not validateMod() then
        Logging.error("[%s] Failed to initialise / validate mod!", modName)

        return
    end

    if not getIsMultiplayerGame() then
        Logging.info("[%s] Singleplayer game detected, mod is designed for MP use only (Locally Hosted or Dedicated Server) and has been disabled.", modName)

        popupMessage = {
            startUpdateTime = 2000,
            update = function(self, dt)
                self.startUpdateTime = self.startUpdateTime - dt

                if self.startUpdateTime < 0 and not g_gui:getIsGuiVisible() and not g_currentMission.hud:isInGameMessageVisible() then
                    if g_currentMission.hud ~= nil then
                        g_currentMission.hud:showInGameMessage("", string.namedFormat(g_i18n:getText("warning_multiplayerNotDetected", modName), "modName", modName), -1, nil, nil, nil)
                    end

                    removeModEventListener(self)
                    popupMessage = nil
                end
            end
        }

        addModEventListener(popupMessage)

        return
    end

    source(modDirectory .. "scripts/events/MultiplayerVehicleKeysSettingsEvent.lua")
    source(modDirectory .. "scripts/events/MultiplayerVehicleKeysSetOwnerEvent.lua")

    g_specializationManager:addSpecialization("multiplayerVehicleKeys", "MultiplayerVehicleKeys", modDirectory .. "scripts/specializations/MultiplayerVehicleKeys.lua", nil)

    g_multiplayerVehicleKeys = MultiplayerVehicleKeysManager.new()
    g_globalMods.multiplayerVehicleKeys = g_multiplayerVehicleKeys

    TypeManager.validateTypes = Utils.prependedFunction(TypeManager.validateTypes, validateTypes)

    BaseMission.loadMapFinished = Utils.appendedFunction(BaseMission.loadMapFinished, loadMapFinished)
    VehicleSystem.delete = Utils.appendedFunction(VehicleSystem.delete, unloadVehicles)

    ItemSystem.save = Utils.prependedFunction(ItemSystem.save, saveItems)
    FSBaseMission.sendInitialClientState = Utils.appendedFunction(FSBaseMission.sendInitialClientState, sendInitialClientState)
end

init()
