--[[
    KeepMeAdmin

    Adds the possibility to keep the admin status on a server.

    @author: 		BayernGamers
    @date: 			09.04.2025
    @version:		1.0

    History:		v1.0 @09.04.2025 - initial implementation in FS 22
                    ------------------------------------------------------------------------------------------------------
    
    License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
                        Attribution:
                            You must give appropriate credit to the original author when using this work.
                        No Derivatives:
                            You may not alter, transform, or build upon this work in any way.
                        Usage: 
                            The work may be used for personal and commercial purposes, provided it is not modified or adapted.
                        Additional Clause:
                            This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/KeepMeAdminSettings.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/utils/EncryptionUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/CustomGetAdminEvent.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/CustomGetAdminAnswerEvent.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/GetMissionDynamicInfoEvent.lua", g_currentModDirectory))

KeepMeAdmin = {}
KeepMeAdmin.MOD_DIR = g_currentModDirectory
KeepMeAdmin.MOD_NAME = g_currentModName
KeepMeAdmin.MOD_SETTINGS_DIR = g_currentModSettingsDirectory
KeepMeAdmin.BASE_PATH = "adminSettings"
KeepMeAdmin.XML_SCHEMA = XMLSchema.new("adminSettings")
KeepMeAdmin.XML_FILENAME = Utils.getFilename("adminSettings.xml", KeepMeAdmin.MOD_SETTINGS_DIR)
KeepMeAdmin.LOGIN_STATE = {
    NONE = 0,
    PENDING = 1,
    GRANTED = 2,
    DENIED = 3
}

local isDedi = g_dedicatedServer ~= nil
local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "KeepMeAdmin.lua")
createFolder(KeepMeAdmin.MOD_SETTINGS_DIR)

function KeepMeAdmin:init()
    log:printDevInfo("Initializing KeepMeAdmin...", LoggingUtil.DEBUG_LEVELS.HIGH)

    local schema = self.XML_SCHEMA

    schema:register(XMLValueType.BOOL, KeepMeAdmin.BASE_PATH .. ".autoLogin#showDialog", "Show dialog for auto login", false)
    schema:register(XMLValueType.BOOL, KeepMeAdmin.BASE_PATH .. ".autoLogin#enabled", "Enable auto login", false)

    local serverKey = KeepMeAdmin.BASE_PATH .. ".servers.server(?)"
    schema:register(XMLValueType.STRING, serverKey .. "#key", "Identifier for the server", nil)
    schema:register(XMLValueType.STRING, serverKey .. "#password", "Password for the server", nil)
end

function KeepMeAdmin.loadMap()
    local self = KeepMeAdmin
    log:printDevInfo("Loading KeepMeAdmin...", LoggingUtil.DEBUG_LEVELS.HIGH)

    if g_currentMission ~= nil and g_currentMission.missionDynamicInfo ~= nil then 
        local missionDynamicInfo = g_currentMission.missionDynamicInfo

        if missionDynamicInfo.isMultiplayer == false then
            log:printWarning("KeepMeAdmin is intended for multiplayer savegames. Please consider removing or deactivating the mod for singleplayer savegames.")
            removeModEventListener(self)
            return
        end

        if missionDynamicInfo.serverAddress == nil then
            log:printDevInfo("Running on a local server. No need to keep the admin status for host.", LoggingUtil.DEBUG_LEVELS.HIGH)
            removeModEventListener(self)
            return
        end
    else
        log:printError("Current mission and/or mission dynamic info is nil. This should not happen. Please report this to the mod author.")
        removeModEventListener(self)
        return
    end

    self.password = nil
    self.autoLogin = KeepMeAdmin.LOGIN_STATE.NONE
    self.settings = KeepMeAdminSettings.new(self)
    self.settings:init()

    g_messageCenter:subscribe(GetAdminAnswerEvent, self.onAdminLoginSuccess, self)
    g_messageCenter:subscribe(CustomGetAdminAnswerEvent, self.onAdminLoginResponse, self)
    g_messageCenter:subscribe(MessageType.CURRENT_MISSION_LOADED, self.onStartMission, self)
end

function KeepMeAdmin:onLoadAdminPassword(yes)
    if yes then
        local password = self:loadServerPassword()

        if password ~= nil then
            self:login(password)
        end
        log:printDevInfo("User accepted to load password.", LoggingUtil.DEBUG_LEVELS.HIGH)
        self:saveAutoLoadPassword(true)
    else
        log:printDevInfo("User declined to load password.", LoggingUtil.DEBUG_LEVELS.HIGH)
        self:saveAutoLoadPassword(false)
    end

    --TODO: Find out where to get inGameMenuMultiplayerFrame from
    local inGameMenuMultiplayerFrame = g_gui.frames["ingameMenuMultiplayer"]
    --g_messageCenter:subscribe(GetAdminAnswerEvent, inGameMenuMultiplayerFrame.onAdminLoginSuccess, inGameMenuMultiplayerFrame)

    local showDialog, _ = self:getUserSettings()
    if showDialog then
        YesNoDialog.show(self.onSaveDialogDefaults, self, g_i18n:getText("ui_confirmationShowLoadAdminPasswordDialog"), g_i18n:getText("ui_showLoadAdminPasswordDialogTitle"))
    end
end

function KeepMeAdmin:onSaveDialogDefaults(yes)
    if yes then
        log:printDevInfo("User accepted to save dialog defaults.", LoggingUtil.DEBUG_LEVELS.HIGH)
        self:saveShowDialog(true)
    else
        log:printDevInfo("User declined to save dialog defaults.", LoggingUtil.DEBUG_LEVELS.HIGH)
        self:saveShowDialog(false)
    end

    --self.blockAutoStart = false
end

function KeepMeAdmin:onSaveServerPassword(yes)
    if yes then
        log:printDevInfo("User accepted to save password.", LoggingUtil.DEBUG_LEVELS.HIGH)
        self:saveServerPassword()
    else
        log:printDevInfo("User declined to save password.", LoggingUtil.DEBUG_LEVELS.HIGH)
        self.password = nil
    end
end

function KeepMeAdmin:onPasswordErrorInfoConfirm()
    log:printDevInfo("User read the error info.", LoggingUtil.DEBUG_LEVELS.HIGH)
    --self.blockAutoStart = false
end

function KeepMeAdmin.onAdminPassword(ingameMultiplayerFrame, password, yes)
    local self = KeepMeAdmin
    log:printDevInfo("Admin password entered: " .. password, LoggingUtil.DEBUG_LEVELS.HIGH)
    self.password = password
end

function KeepMeAdmin:onAdminLoginSuccess()
    log:printDevInfo("Received admin login success event.", LoggingUtil.DEBUG_LEVELS.HIGH)
    if self.autoLogin == self.LOGIN_STATE.NONE or self.autoLogin == KeepMeAdmin.LOGIN_STATE.DENIED then
        log:printDevInfo("Admin login granted.", LoggingUtil.DEBUG_LEVELS.HIGH)

        YesNoDialog.show(self.onSaveServerPassword, self, g_i18n:getText("ui_confirmationKeepAdminPasword"), g_i18n:getText("ui_keepAdminPasswordTitle"))
    end
end

function KeepMeAdmin:onAdminLoginResponse(state)
    log:printDevInfo("Admin login response received with state: " .. tostring(state), LoggingUtil.DEBUG_LEVELS.HIGH)

    if state == CustomGetAdminAnswerEvent.NOT_SUPPORTED then
        log:printWarning("The requested Login returned NOT_SUPPORTED.")
        return
    end

    if self.autoLogin == self.LOGIN_STATE.PENDING then
        if state == CustomGetAdminAnswerEvent.ACCESS_GRANTED then
            log:printDevInfo("Pending Admin login granted.", LoggingUtil.DEBUG_LEVELS.HIGH)
            self.autoLogin = KeepMeAdmin.LOGIN_STATE.GRANTED
        elseif state == CustomGetAdminAnswerEvent.ACCESS_DENIED then
            log:printDevInfo("Pending Admin login denied.", LoggingUtil.DEBUG_LEVELS.HIGH)
            self.autoLogin = KeepMeAdmin.LOGIN_STATE.DENIED
            --self.blockAutoStart = true
            self:deleteServerPassword()

            InfoDialog.show(g_i18n:getText("ui_infoWrongStoredPassword"), self.onPasswordErrorInfoConfirm, self, DialogElement.TYPE_WARNING)
        end
    end

    if self.autoLogin == self.LOGIN_STATE.GRANTED then
        log:printDevInfo("Publishing GetAdminAnswerEvent", LoggingUtil.DEBUG_LEVELS.HIGH)
        --g_messageCenter:publish(GetAdminAnswerEvent, true)

        --local userManager = g_currentMission.userManager
        --userManager:addMasterUser(userManager:getUserByUserId(g_localPlayer.userId))
        --g_client:getServerConnection():sendEvent(GetMissionDynamicInfoEvent.new())

        log:printDevInfo("Current Mission UserManager: " .. tostring(g_currentMission.userManager), LoggingUtil.DEBUG_LEVELS.HIGH)
        log:printDevInfo("Number of Master Users: " .. tostring(g_currentMission.userManager:getNumberOfMasterUsers()), LoggingUtil.DEBUG_LEVELS.HIGH)
    end
end

function KeepMeAdmin:getXMLFile()
    local created = false
    local xmlFile = XMLFile.loadIfExists("adminSettings", self.MOD_SETTINGS_DIR .. "adminSettings.xml", self.XML_SCHEMA)

    if xmlFile == nil then
        log:printDevInfo("No adminSettings.xml found. Creating new one.", LoggingUtil.DEBUG_LEVELS.HIGH)
        xmlFile = XMLFile.create("adminSettings", self.XML_FILENAME, "adminSettings", self.XML_SCHEMA)
        created = true
    end

    if xmlFile == nil then
        if created then
            log:printError("Failed to create adminSettings.xml. Please check your permissions.", LoggingUtil.DEBUG_LEVELS.HIGH)
        else
            log:printError("Failed to load adminSettings.xml. Please check your permissions.", LoggingUtil.DEBUG_LEVELS.HIGH)
        end
    end

    return xmlFile
end

function KeepMeAdmin:getUserSettings()
    local xmlFile = self:getXMLFile()

    if xmlFile == nil then
        return nil, nil
    end

    local showDialog = xmlFile:getValue(KeepMeAdmin.BASE_PATH .. ".autoLogin#showDialog", true)
    local enabled = xmlFile:getValue(KeepMeAdmin.BASE_PATH .. ".autoLogin#enabled", false)

    xmlFile:delete()
    return showDialog, enabled
end

function KeepMeAdmin:saveAutoLoadPassword(enabled)
    local xmlFile = self:getXMLFile()

    if xmlFile == nil then
        return
    end

    if enabled == nil then
        enabled = false
    end

    xmlFile:setValue(KeepMeAdmin.BASE_PATH .. ".autoLogin#enabled", enabled)
    xmlFile:save()
    xmlFile:delete()
end

function KeepMeAdmin:saveShowDialog(showDialog)
    local xmlFile = self:getXMLFile()

    if xmlFile == nil then
        return
    end

    if showDialog == nil then
        showDialog = false
    end

    xmlFile:setValue(KeepMeAdmin.BASE_PATH .. ".autoLogin#showDialog", showDialog)
    xmlFile:save()
    xmlFile:delete()
end

function KeepMeAdmin:getServerKey()
    if g_currentMission ~= nil and g_currentMission.missionDynamicInfo == nil then
        log:printError("Global dedicated server variable does not exist. Are you in multiplayer?", LoggingUtil.DEBUG_LEVELS.HIGH)
        return nil
    end

    local missionDynamicInfo = g_currentMission.missionDynamicInfo

    log:printDevInfo("Mission dynamic info: ", LoggingUtil.DEBUG_LEVELS.HIGH)
    for k, v in pairs(missionDynamicInfo) do
        log:printDevInfo(k .. ": " .. tostring(v), LoggingUtil.DEBUG_LEVELS.HIGH)
    end

    local serverId = getMD5(missionDynamicInfo.serverAddress .. ":" .. missionDynamicInfo.serverPort)
    log:printDevInfo("Retreived Server ID: " .. tostring(serverId), LoggingUtil.DEBUG_LEVELS.HIGH)
    return serverId
end

function KeepMeAdmin:loadServerPassword()
    local xmlFile = self:getXMLFile()
    if xmlFile == nil then
        return nil
    end

    local serverKey = self:getServerKey()
    if serverKey == nil then
        return nil
    end

    local password = nil
    xmlFile:iterate("adminSettings.servers.server", function (i, key)
        if xmlFile:getValue(key .. "#key") == serverKey then
            local encryptedPassword = xmlFile:getValue(key .. "#password")
            log:printDevInfo("Found password for server " .. serverKey .. ": " .. tostring(encryptedPassword), LoggingUtil.DEBUG_LEVELS.HIGH)

            password = EncryptionUtil.decrypt(encryptedPassword, g_currentMission.missionDynamicInfo.password)
            log:printDevInfo("Decrypted password: " .. password, LoggingUtil.DEBUG_LEVELS.HIGH)
            return false
        end
    end)

    xmlFile:delete()
    return password
end

function KeepMeAdmin:saveServerPassword()
    local xmlFile = self:getXMLFile()
    local serverKey = self:getServerKey()
    -- TODO: Handle xmlFile == nil and serverKey == nil
    
    log:printDevInfo("Encrypting password: " .. tostring(self.password), LoggingUtil.DEBUG_LEVELS.HIGH)
    local encryptedPassword = EncryptionUtil.encrypt(self.password, g_currentMission.missionDynamicInfo.password)
    log:printDevInfo("Encrypted password: " .. tostring(encryptedPassword), LoggingUtil.DEBUG_LEVELS.HIGH)

    local lastIndex = 0
    local found = false
    local stored = false
    xmlFile:iterate("adminSettings.servers.server", function (i, key)
        lastIndex = i

        if xmlFile:getValue(key .. "#key") == serverKey then
            xmlFile:setValue(key .. "#password", encryptedPassword)
            log:printDevInfo("Saved password for server " .. serverKey .. ": " .. self.password, LoggingUtil.DEBUG_LEVELS.HIGH)
            found = true
            stored = true
            return false
        end
    end)

    if not found then
        local newKey = string.format("adminSettings.servers.server(%d)", lastIndex)
        xmlFile:setValue(newKey .. "#key", serverKey)
        xmlFile:setValue(newKey .. "#password", encryptedPassword)
        log:printDevInfo("Saved new password for server " .. serverKey .. ": " .. encryptedPassword, LoggingUtil.DEBUG_LEVELS.HIGH)
        stored = true
    end

    if stored then
        xmlFile:save()
        self.password = nil
        log:printDevInfo("Saved adminSettings.xml.", LoggingUtil.DEBUG_LEVELS.HIGH)
    else
        log:printXMLWarning(xmlFile, "Failed to save adminSettings.xml.")
    end

    xmlFile:delete()
end

function KeepMeAdmin:deleteServerPassword()
    local xmlFile = self:getXMLFile()
    local serverKey = self:getServerKey()

    xmlFile:iterate("adminSettings.servers.server", function (i, key)
        if xmlFile:getValue(key .. "#key") == serverKey then
            xmlFile:removeProperty(key)
            log:printDevInfo("Deleted password for server " .. serverKey, LoggingUtil.DEBUG_LEVELS.HIGH)
            xmlFile:save()
            return false
        end
    end)

    xmlFile:delete()
end

function KeepMeAdmin:login(password)
    log:printDevInfo("Logging in with password: " .. password, LoggingUtil.DEBUG_LEVELS.HIGH)
    self.autoLogin = KeepMeAdmin.LOGIN_STATE.PENDING

    g_client:getServerConnection():sendEvent(CustomGetAdminEvent.new(password))
end

function KeepMeAdmin:onStartMission()
    local showDialog, enabled = self:getUserSettings()

    if showDialog then
        --self.blockAutoStart = true
        YesNoDialog.show(self.onLoadAdminPassword, self, g_i18n:getText("ui_confirmationLoadStoredAdminPassword"), g_i18n:getText("ui_loadStoredAdminPasswordTitle"))
    elseif enabled then
        self:onLoadAdminPassword(true)
    else
        local inGameMenuMultiplayerFrame = g_gui.frames["ingameMenuMultiplayer"]
        --g_messageCenter:subscribe(GetAdminAnswerEvent, inGameMenuMultiplayerFrame.onAdminLoginSuccess, inGameMenuMultiplayerFrame)
    end
end

if not isDedi then
    KeepMeAdmin:init()
    addModEventListener(KeepMeAdmin)
    InGameMenuMultiplayerFrame.onAdminPassword = Utils.appendedFunction(InGameMenuMultiplayerFrame.onAdminPassword, KeepMeAdmin.onAdminPassword)
end