-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

local g_thModPath = g_currentModDirectory
source(g_thModPath .. "shared/scripts/THCore.lua")
source(g_thModPath .. "shared/scripts/managers/THSpecManager.lua")
THBetterWinch = {}
local THBetterWinch_mt = Class(THBetterWinch, THCore)
THBetterWinch.NAME = "thBetterWinch"
THBetterWinch.CLASS_NAME = "THBetterWinch"
THBetterWinch.DEFAULT_ATTACH_RADIUS = 0.1
THBetterWinch.MIN_ATTACH_RADIUS = 0.01
THBetterWinch.MAX_ATTACH_RADIUS = 10
THBetterWinch.ATTACH_RADIUS_ACCEL_DELAY = 1000
THBetterWinch.COLLISION_MASK = CollisionFlag.DYNAMIC_OBJECT + CollisionFlag.VEHICLE + CollisionFlag.TRIGGER
local debugFlagId = THCore.debugFlagId
local function initScript()
    local self = THBetterWinch.new(THBetterWinch.NAME)
    if self ~= nil then
        _G.g_thBetterWinch = self
        THUtils.overwriteFunction(g_thGlobalEnv, "readSplitShapeIdFromStream", false, false, self, THBetterWinch.overwrite_readSplitShapeIdFromStream)
        THUtils.overwriteFunction(g_thGlobalEnv, "writeSplitShapeIdToStream", false, false, self, THBetterWinch.overwrite_writeSplitShapeIdToStream)
        THUtils.overwriteFunction("SplitShapeUtil", "getTreeOffsetPosition", false, false, self, THBetterWinch.overwrite_getTreeOffsetPosition)
        THUtils.overwriteFunction("SplitShapeUtil", "createTreeBelt", false, false, self, THBetterWinch.overwrite_createSplitShapeTreeBelt)
    end
end
function THBetterWinch.new(name, customMt)
    customMt = customMt or THBetterWinch_mt
    local self = THCore.new(name, customMt)
    if self ~= nil then
        self.objectsI3DFilename = "data/injectedAttachPoints.i3d"
        self.attachableObjects = {
            byIndex = {},
            byTarget = {},
            byComponentNode = {},
            byMeshNode = {}
        }
        self.objectAttachPoints = {}
        self.clonedAttachPoints = {}
        self.attachPointNodes = {}
        self.attachPointRootNodes = {}
        return self
    end
end
function THBetterWinch.onPreInit(self, ...)
    THUtils.callSuperClass(self, "onPreInit", true, ...)
    local modPath = self.coreData.mod.path
    source(modPath .. "scripts/objects/THBetterWinchAttachable.lua")
end
function THBetterWinch.onLoadMapFinished(self, mission, ...)
    THUtils.callSuperClass(self, "onLoadMapFinished", true, mission, ...)
    local modPath = self.coreData.mod.path
    local objectsI3DFilename = THUtils.getFilename(self.objectsI3DFilename, modPath)
    local objectsRootNode, sharedLoadRequestId = g_i3DManager:loadSharedI3DFile(objectsI3DFilename, false, false)
    local rootNode = getRootNode()
    local attachNode = getChildAt(objectsRootNode, 0)
    removeFromPhysics(attachNode)
    link(rootNode, attachNode)
    self.attachPointTemplate = attachNode
    self.sharedLoadRequestId = sharedLoadRequestId
end
function THBetterWinch.onDelete(self, ...)
    THUtils.callSuperClass(self, "onDelete", true, ...)
    local mission = self.coreData.mission
    for attachPointNode in pairs(self.attachPointNodes) do
        mission:removeNodeObject(attachPointNode)
        self.attachPointNodes[attachPointNode] = nil
    end
    THUtils.clearTable(self.objectAttachPoints)
    THUtils.clearTable(self.attachPointRootNodes)
    for attachPointIndex = #self.clonedAttachPoints, 1, -1 do
        local attachNode = self.clonedAttachPoints[attachPointIndex]
        if THUtils.getNodeExists(attachNode) then
            unlink(attachNode)
            delete(attachNode)
        end
        table.remove(self.clonedAttachPoints, attachPointIndex)
    end
    if THUtils.getNodeExists(self.attachPointTemplate) then
        unlink(self.attachPointTemplate)
        delete(self.attachPointTemplate)
        self.attachPointTemplate = nil
    end
    if self.sharedLoadRequestId ~= nil then
        g_i3DManager:releaseSharedI3DFile(self.sharedLoadRequestId)
        self.sharedLoadRequestId = nil
    end
end
function THBetterWinch.getObjectComponentByNodeId(self, object, nodeId)
    local mission = self.coreData.mission
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(nodeId, THValueType.INTEGER) and nodeId > 0, "nodeId", nodeId)
        and mission ~= nil
    then
        local nodeName = nil
        if THUtils.getNodeExists(nodeId) then
            nodeName = getName(nodeId)
            if type(object.components) == THValueType.TABLE and type(object.components[1]) == THValueType.TABLE then
                for componentIndex = #object.components, 1, -1 do
                    local componentInfo = object.components[componentIndex]
                    if I3DUtil.getIsLinkedToNode(componentInfo.node, nodeId) then
                        return componentIndex, componentInfo.node
                    end
                end
            elseif type(object.nodeId) == THValueType.NUMBER and object.nodeId > 0 then
                if I3DUtil.getIsLinkedToNode(object.nodeId, nodeId) then
                    return 1, object.nodeId
                end
            elseif type(object.rootNode) == THValueType.NUMBER and object.rootNode > 0 then
                if I3DUtil.getIsLinkedToNode(object.rootNode, nodeId) then
                    return 1, object.rootNode
                end
            end
        end
        nodeName = nodeName or "unknown"
        THUtils.printDebugMsg(nil, nil, "Could not find component index for node: %s [%s]", nodeName, nodeId)
    end
end
function THBetterWinch.getObjectComponentByIndex(self, object, componentIndex)
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(componentIndex, THValueType.INTEGER), "componentIndex", componentIndex)
    then
        if type(object.components) == THValueType.TABLE then
            local componentInfo = object.components[componentIndex]
            if componentInfo ~= nil then
                if THUtils.getIsType(componentInfo.node, THValueType.INTEGER) and componentInfo.node > 0 then
                    return componentInfo.node
                end
                THUtils.printDebugMsg(nil, nil, "Invalid component [%s] node: %s", componentIndex, componentInfo.node)
            else
                THUtils.printDebugMsg(nil, nil, THMessage.ARGUMENT_INVALID, "componentIndex", componentIndex)
            end
        end
        if componentIndex == 1 then
            if THUtils.getIsType(object.nodeId, THValueType.INTEGER) and object.nodeId > 0 then
                return object.nodeId
            end
            if THUtils.getIsType(object.rootNode, THValueType.INTEGER) and object.rootNode > 0 then
                return object.rootNode
            end
            THUtils.printDebugMsg(nil, nil, "Could not find object component node")
        else
            THUtils.printDebugMsg(nil, nil, THMessage.ARGUMENT_INVALID, "componentIndex", componentIndex)
        end
    end
end
function THBetterWinch.getObjectSize(self, object)
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object) then
        local sizeX, sizeY, sizeZ = nil, nil, nil
        if type(object.size) == THValueType.TABLE then
            if type(object.size.width) == THValueType.NUMBER and object.size.width > 0 then
                sizeX = object.size.width
            end
            if type(object.size.height) == THValueType.NUMBER and object.size.height > 0 then
                sizeY = object.size.height
            end
            if type(object.size.length) == THValueType.NUMBER and object.size.length > 0 then
                sizeZ = object.size.length
            end
        end
        if sizeX == nil
            and type(object.width) == THValueType.NUMBER and object.width > 0
        then
            sizeX = object.width
        end
        if sizeY == nil
            and type(object.height) == THValueType.NUMBER and object.height > 0
        then
            sizeY = object.height
        end
        if sizeZ == nil
            and type(object.length) == THValueType.NUMBER and object.length > 0
        then
            sizeZ = object.length
        end
        if type(object.diameter) == THValueType.NUMBER and object.diameter > 0 then
            sizeX = sizeX or object.diameter
            sizeY = sizeY or object.diameter
            sizeZ = sizeZ or object.diameter
        end
        if sizeX == nil then
            if sizeZ ~= nil then
                if sizeY ~= nil then
                    sizeX = math.max(sizeY, sizeZ)
                else
                    sizeX = sizeZ
                end
            else
                sizeZ = THBetterWinch.DEFAULT_ATTACH_RADIUS * 2
            end
        end
        if sizeY == nil then
            if sizeZ ~= nil then
                if sizeX ~= nil then
                    sizeY = math.max(sizeX, sizeZ)
                else
                    sizeY = sizeZ
                end
            else
                sizeY = THBetterWinch.DEFAULT_ATTACH_RADIUS * 2
            end
        end
        if sizeZ == nil then
            if sizeX ~= nil then
                if sizeY ~= nil then
                    sizeZ = math.max(sizeX, sizeY)
                else
                    sizeZ = sizeX
                end
            else
                sizeZ = THBetterWinch.DEFAULT_ATTACH_RADIUS * 2
            end
        end
        return sizeX, sizeY, sizeZ
    end
end
function THBetterWinch.addObjectAttachPoint(self, object, nodeId)
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(nodeId, THValueType.INTEGER) and nodeId > 0, "nodeId", nodeId)
    then
        local mission = self.coreData.mission
        if not THUtils.getNodeExists(nodeId) then
            THUtils.errorMsg(nil, "Cannot find attach point node: %s", nodeId)
        elseif not getHasClassId(nodeId, ClassIds.SHAPE) then
            THUtils.errorMsg(nil, "Attach point must be a valid shape")
        elseif not getShapeIsCPUMesh(nodeId) then
            THUtils.errorMsg(nil, "Attach point must have the cpu mesh flag set")
        else
            local hasUserAttribute = getUserAttribute(nodeId, "thAttachPoint")
            local isNodeIdValid = true
            if getHasCollision(nodeId) then
                local rigidBodyType = getRigidBodyType(nodeId)
                if rigidBodyType == RigidBodyType.KINEMATIC then
                    if getCollisionFilterGroup(nodeId) ~= CollisionFlag.TRIGGER then
                        if not hasUserAttribute then
                            THUtils.errorMsg(nil, "Kinematic rigid body attach points must have only filter group TRIGGER [29] set")
                        end
                        isNodeIdValid = false
                    end
                elseif rigidBodyType ~= RigidBodyType.DYNAMIC then
                    THUtils.errorMsg(nil, "Rigid body attach points must be either kinematic or dynamic")
                    isNodeIdValid = false
                end
                if not isNodeIdValid and hasUserAttribute
                    and rigidBodyType == RigidBodyType.KINEMATIC
                then
                    removeFromPhysics(nodeId)
                    setCollisionFilterGroup(nodeId, CollisionFlag.TRIGGER)
                    setCollisionFilterMask(nodeId, CollisionFlag.PLAYER)
                    isNodeIdValid = true
                end
            end
            if isNodeIdValid then
                local attachPoints = self.objectAttachPoints[object]
                local isAttachPointFound = false
                if attachPoints == nil then
                    attachPoints = {}
                    self.objectAttachPoints[object] = attachPoints
                else
                    for _, attachPointData in pairs(attachPoints) do
                        if attachPointData.node == nodeId then
                            isAttachPointFound = true
                            break
                        end
                    end
                end
                if not isAttachPointFound then
                    local componentIndex, componentNode = self:getObjectComponentByNodeId(object, nodeId)
                    if componentIndex ~= nil and componentIndex > 0
                        and componentNode ~= nil and componentNode > 0
                    then
                        local attachPointData = {
                            object = object,
                            node = nodeId,
                            nodeName = getName(nodeId),
                            rootNode = componentNode,
                            component = componentIndex
                        }
                        local otherNodeObject = mission.nodeToObject[attachPointData.node]
                        if otherNodeObject == nil or otherNodeObject == object then
                            if otherNodeObject == nil then
                                mission:addNodeObject(attachPointData.node, attachPointData.object)
                            end
                            table.insert(attachPoints, attachPointData)
                            self.attachPointNodes[attachPointData.node] = attachPointData.node
                            self.attachPointRootNodes[attachPointData.node] = attachPointData.rootNode
                            return attachPointData
                        end
                        THUtils.errorMsg(nil, "Attach point node %s [%s] already assigned to another object", attachPointData.nodeName, attachPointData.node)
                    end
                end
            end
        end
    end
end
function THBetterWinch.removeObjectAttachPoint(self, object, nodeId)
    local mission = self.coreData.mission
    local success = false
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(nodeId, THValueType.INTEGER) and nodeId > 0, "nodeId", nodeId)
    then
        if mission ~= nil then
            local attachPointsArray, numAttachPoints = self:getObjectAttachPoints(object)
            if attachPointsArray ~= nil and numAttachPoints > 0 then
                for attachPointIndex = numAttachPoints, 1, -1 do
                    local attachPointData = attachPointsArray[attachPointIndex]
                    if attachPointData.node == nodeId then
                        mission:removeNodeObject(nodeId)
                        if self.attachPointNodes[attachPointData.node] ~= nil then
                            self.attachPointNodes[attachPointData.node] = nil
                        end
                        if self.attachPointRootNodes[attachPointData.node] ~= nil then
                            self.attachPointRootNodes[attachPointData.node] = nil
                        end
                        table.remove(attachPointsArray, attachPointIndex)
                        if THUtils.getIsDebugEnabled(debugFlagId) then
                            local nodeName = nil
                            if THUtils.getNodeExists(nodeId) then
                                nodeName = getName(nodeId)
                            end
                            nodeName = nodeName or "unknown"
                            THUtils.objectDisplayMsg(object, "Removed attach point: %s [%s]", nodeName, nodeId)
                        end
                        success = true
                    end
                end
            end
        end
    end
    return success
end
function THBetterWinch.injectObjectAttachPoint(self, object, parentNode, offsetX, offsetY, offsetZ, rotX, rotY, rotZ)
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(parentNode, THValueType.INTEGER) and parentNode > 0, "parentNode", parentNode)
        and THUtils.argIsValid(type(offsetX) == THValueType.NUMBER, "offsetX", offsetX)
        and THUtils.argIsValid(type(offsetY) == THValueType.NUMBER, "offsetY", offsetY)
        and THUtils.argIsValid(type(offsetZ) == THValueType.NUMBER, "offsetZ", offsetZ)
        and THUtils.argIsValid(type(rotX) == THValueType.NUMBER, "rotX", rotX)
        and THUtils.argIsValid(type(rotY) == THValueType.NUMBER, "rotY", rotY)
        and THUtils.argIsValid(type(rotZ) == THValueType.NUMBER, "rotZ", rotZ)
    then
        if not THUtils.getNodeExists(parentNode) then
            THUtils.errorMsg(nil, "Can not find attach point parent node: %s", parentNode)
        else
            local attachPoints = self.objectAttachPoints[object]
            local isAttachPointFound = false
            if attachPoints == nil then
                attachPoints = {}
                self.objectAttachPoints[object] = attachPoints
            else
                for _, attachPointData in pairs(attachPoints) do
                    if attachPointData.parentNode ~= nil
                        and attachPointData.parentNode == parentNode
                    then
                        isAttachPointFound = true
                        break
                    end
                end
            end
            if not isAttachPointFound then
                local attachPointNode = clone(self.attachPointTemplate, false, false, false)
                if attachPointNode ~= nil and attachPointNode > 0 then
                    local hasCollision = getHasCollision(attachPointNode)
                    if hasCollision then
                        removeFromPhysics(attachPointNode)
                    end
                    link(parentNode, attachPointNode)
                    setTranslation(attachPointNode, offsetX, offsetY, offsetZ)
                    setRotation(attachPointNode, rotX, rotY, rotZ)
                    local attachPointData = self:addObjectAttachPoint(object, attachPointNode)
                    if attachPointData ~= nil then
                        attachPointData.parentNode = parentNode
                        table.insert(self.clonedAttachPoints, attachPointNode)
                        return attachPointData
                    end
                    delete(attachPointNode)
                end
            end
        end
    end
end
function THBetterWinch.removeObjectInjectedAttachPoint(self, object, nodeId)
    local success = false
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object)
        and THUtils.argIsValid(THUtils.getIsType(nodeId, THValueType.INTEGER) and nodeId > 0, "nodeId", nodeId)
    then
        local numClonedAttachPoints = #self.clonedAttachPoints
        if numClonedAttachPoints > 0 then
            for attachPointIndex = numClonedAttachPoints, 1, -1 do
                local attachPointNode = self.clonedAttachPoints[attachPointIndex]
                if attachPointNode == nodeId then
                    local nodeName = nil
                    self:removeObjectAttachPoint(object, nodeId)
                    if THUtils.getNodeExists(nodeId) then
                        nodeName = getName(nodeId)
                        delete(nodeId)
                    end
                    table.remove(self.clonedAttachPoints, attachPointIndex)
                    if THUtils.getIsDebugEnabled(debugFlagId) then
                        nodeName = nodeName or "unknown"
                        THUtils.objectDisplayMsg(object, "Removed injected attach point: %s [%s]", nodeName, nodeId)
                    end
                    success = true
                end
            end
        end
    end
    return success
end
function THBetterWinch.getObjectAttachPoints(self, object)
    if object ~= nil then
        local attachPointsArray = self.objectAttachPoints[object]
        if attachPointsArray ~= nil then
            return attachPointsArray, #attachPointsArray
        end
    end
    return nil, 0
end
function THBetterWinch.getAttachPointNode(self, nodeId)
    if nodeId ~= nil then
        return self.attachPointNodes[nodeId]
    end
end
function THBetterWinch.getAttachPointRootNode(self, nodeId)
    if nodeId ~= nil then
        return self.attachPointRootNodes[nodeId]
    end
end
function THBetterWinch.validateAttachableObjectNodeId(self, nodeId, verbose)
    local mission = self.coreData.mission
    verbose = THUtils.validateArg(not verbose or verbose == true, "verbose", verbose, false)
    if type(nodeId) ~= THValueType.NUMBER or nodeId <= 0 then
        if verbose then
            THUtils.printDebugError(nil, true, THMessage.ARGUMENT_INVALID, "nodeId", nodeId)
        end
    elseif not THUtils.getNodeExists(nodeId) then
        if verbose then
            THUtils.printDebugError(nil, nil, "Component node %q does not exist", nodeId)
        end
    else
        local nodeName = getName(nodeId)
        if not getHasClassId(nodeId, ClassIds.SHAPE) then
            if verbose then
                THUtils.printDebugError(nil, nil, "Component node %q [%s] is not a valid shape", nodeName, nodeId)
            end
        elseif not getHasCollision(nodeId) then
            if verbose then
                THUtils.printDebugError(nil, nil, "Component node %q [%s] must be a rigid body shape", nodeName, nodeId)
            end
        elseif not CollisionFlag.getHasGroupFlagSet(nodeId, THBetterWinch.COLLISION_MASK) then
            if verbose then
                THUtils.printDebugError(nil, nil, "Component node %q [%s] does not have proper collision group set", nodeName, nodeId)
            end
        else
            local splitType = getSplitType(nodeId)
            if type(splitType) == THValueType.NUMBER and splitType > 0 then
                if verbose then
                    THUtils.printDebugError(nil, nil, "Component node %q [%s] is a tree and covered by the vanilla system", nodeName, nodeId)
                end
            elseif mission == nil then
                if verbose then
                    THUtils.printDebugError(nil, true, "Mission table does not exist")
                end
            else
                local object = mission:getNodeObject(nodeId)
                if THUtils.getIsType(object, Object) then
                    return true
                end
                if verbose then
                    THUtils.printDebugError(nil, nil, "No object associated with component node %q [%s]", nodeName, nodeId)
                end
            end
        end
    end
    return false
end
function THBetterWinch.getAttachableObject(self, object, force)
    if THUtils.argIsValid(not force or force == true, "force", force) then
        if object ~= nil then
            local objectData = self.attachableObjects.byTarget[object]
            if objectData ~= nil then
                if objectData:getIsEnabled() or force then
                    return objectData
                end
            end
        end
    end
end
function THBetterWinch.getAttachableObjectByIndex(self, index, force)
    if THUtils.argIsValid(not force or force == true, "force", force) then
        if index ~= nil then
            local objectData = self.attachableObjects.byIndex[index]
            if objectData ~= nil then
                if objectData:getIsEnabled() or force then
                    return objectData
                end
            end
        end
    end
end
function THBetterWinch.getAttachableObjectByComponentNode(self, nodeId, force)
    if THUtils.argIsValid(not force or force == true, "force", force) then
        if nodeId ~= nil then
            local objectData = self.attachableObjects.byComponentNode[nodeId]
            if objectData ~= nil then
                if objectData:getIsEnabled() or force then
                    return objectData
                end
            end
        end
    end
end
function THBetterWinch.getAttachableObjectByMeshNode(self, nodeId, force)
    if THUtils.argIsValid(not force or force == true, "force", force) then
        if nodeId ~= nil then
            local objectData = self.attachableObjects.byMeshNode[nodeId]
            if objectData ~= nil then
                if objectData:getIsEnabled() or force then
                    return objectData
                end
            end
        end
    end
end
function THBetterWinch.getAttachableObjectByNodeId(self, nodeId, force)
    if THUtils.argIsValid(not force or force == true, "force", force) then
        if nodeId ~= nil then
            local objectData = self:getAttachableObjectByMeshNode(nodeId, true)
            local isMeshNode = false
            if objectData ~= nil then
                isMeshNode = true
            else
                objectData = self:getAttachableObjectByComponentNode(nodeId, true)
            end
            if objectData ~= nil then
                if objectData:getIsEnabled() or force then
                    return objectData, isMeshNode
                end
            end
        end
    end
    return nil, false
end
function THBetterWinch.registerAttachableObject(self, object)
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object) then
        local objectData = self:getAttachableObject(object, true)
        if objectData ~= nil then
            return objectData
        end
        objectData = THBetterWinchAttachable.new(object)
        if objectData ~= nil then
            if objectData:load() then
                objectData.index = #self.attachableObjects.byIndex + 1
                self.attachableObjects.byIndex[objectData.index] = objectData
                self.attachableObjects.byTarget[objectData.target] = objectData
                local componentsArray, numComponents = objectData:getComponents()
                if componentsArray ~= nil and numComponents > 0 then
                    for _, componentData in pairs(componentsArray) do
                        local otherObjectData = self.attachableObjects.byComponentNode[componentData.node]
                        if otherObjectData ~= nil and otherObjectData ~= objectData then
                            THUtils.printDebugMsg(nil, nil, "Component %q [%s] is registered to another object", componentData.nodeName, componentData.node)
                        end
                        self.attachableObjects.byComponentNode[componentData.node] = objectData
                    end
                end
                local meshNodesArray, numMeshNodes = objectData:getMeshNodes()
                if meshNodesArray ~= nil and numMeshNodes > 0 then
                    for _, meshNodeData in pairs(meshNodesArray) do
                        local otherObjectData = self.attachableObjects.byMeshNode[meshNodeData.node]
                        if otherObjectData ~= nil and otherObjectData ~= objectData then
                            THUtils.printDebugMsg(nil, nil, "Mesh node %q [%s] is registered to another object", meshNodeData.nodeName, meshNodeData.node)
                        end
                        self.attachableObjects.byMeshNode[meshNodeData.node] = objectData
                    end
                end
                if THUtils.getIsDebugEnabled(debugFlagId) then
                    THUtils.displayMsg("Added object:")
                    THUtils.printTable(objectData)
                    THUtils.displayMsg("Component nodes:")
                    THUtils.printTable(objectData.components, 3)
                    THUtils.displayMsg("Mesh nodes:")
                    THUtils.printTable(objectData.meshNodes, 3)
                end
                return objectData
            end
        end
    end
end
function THBetterWinch.unregisterAttachableObject(self, object)
    local success = false
    if THUtils.argIsValid(THUtils.getIsType(object, Object), "object", object) then
        local objectData = self:getAttachableObject(object, true)
        if objectData ~= nil then
            local componentsArray, numComponents = objectData:getComponents()
            if componentsArray ~= nil and numComponents > 0 then
                for _, componentData in pairs(componentsArray) do
                    local otherObjectData = self.attachableObjects.byComponentNode[componentData.node]
                    if otherObjectData == objectData then
                        self.attachableObjects.byComponentNode[componentData.node] = nil
                    end
                end
            end
            local meshNodesArray, numMeshNodes = objectData:getMeshNodes()
            if meshNodesArray ~= nil and numMeshNodes > 0 then
                for _, meshNodeData in pairs(meshNodesArray) do
                    local otherObjectData = self.attachableObjects.byMeshNode[meshNodeData.node]
                    if otherObjectData == objectData then
                        self.attachableObjects.byMeshNode[meshNodeData.node] = nil
                    end
                end
            end
            local numAttachableObjects = #self.attachableObjects.byIndex
            if numAttachableObjects > 0 then
                local objectIndex = 1
                while true do
                    local otherObjectData = self.attachableObjects.byIndex[objectIndex]
                    if otherObjectData == nil then
                        break
                    end
                    if otherObjectData == objectData then
                        table.remove(self.attachableObjects.byIndex, objectIndex)
                    else
                        otherObjectData.index = objectIndex
                        objectIndex = objectIndex + 1
                    end
                end
            end
            self.attachableObjects.byTarget[objectData.target] = nil
            objectData:delete()
            if THUtils.getIsDebugEnabled(debugFlagId) then
                THUtils.objectDisplayMsg(object, "Object unregistered")
            end
            success = true
        end
    end
    return success
end
function THBetterWinch.getAttachableObjectOffsetPosition(self, nodeId, x, y, z, maxRadius, minLength)
    local objectData, isMeshNode = self:getAttachableObjectByNodeId(nodeId)
    if objectData ~= nil then
        local meshData = nil
        if isMeshNode then
            meshData = objectData:getMeshNode(nodeId, false, true)
        else
            local _ = nil
            _, meshData = objectData:getLastTargetData()
            if meshData == nil then
                meshData = objectData:getClosestMeshNode(x, y, z, maxRadius)
            end
        end
        if meshData ~= nil then
            local cx, cy, cz = getWorldTranslation(meshData.node)
            local upX, upY, upZ = nil, nil, nil
            if meshData.upAxis == THAxis.X then
                upX, upY, upZ = localDirectionToWorld(meshData.node, 1, 0, 0)
            elseif meshData.upAxis == THAxis.Z then
                upX, upY, upZ = localDirectionToWorld(meshData.node, 0, 0, 1)
            else
                upX, upY, upZ = localDirectionToWorld(meshData.node, 0, 1, 0)
            end
            if cx ~= nil and upX ~= nil then
                local radius = meshData.radius
                if radius == nil or radius < THBetterWinch.MIN_ATTACH_RADIUS then
                    if meshData.upAxis == THAxis.X then
                        radius = math.max(meshData.size[THAxis.Y], meshData.size[THAxis.Z]) * 0.5
                    elseif meshData.upAxis == THAxis.Z then
                        radius = math.max(meshData.size[THAxis.X], meshData.size[THAxis.Y]) * 0.5
                    else
                        radius = math.max(meshData.size[THAxis.X], meshData.size[THAxis.Z]) * 0.5
                    end
                    if radius == nil or radius < THBetterWinch.MIN_ATTACH_RADIUS then
                        radius = THBetterWinch.DEFAULT_ATTACH_RADIUS
                    end
                end
                radius = THUtils.clamp(radius, THBetterWinch.MIN_ATTACH_RADIUS, THBetterWinch.MAX_ATTACH_RADIUS)
                if meshData.radius == nil then
                    meshData.radius = radius
                end
                if maxRadius ~= nil and maxRadius > 0 then
                    radius = math.min(radius, maxRadius)
                end
                if THUtils.getIsDebugEnabled(debugFlagId, THDebugLevel.UPDATE) then
                    THUtils.displayMsg("Updating object offset position:")
                    THUtils.displayMsg("- Radius: %s", radius)
                    THUtils.displayMsg("- cx, cy, cz: %0.3f, %0.3f, %0.3f", cx, cy, cz)
                    THUtils.displayMsg("- upX, upY, upZ: %0.3f, %0.3f, %0.3f", upX, upY, upZ)
                    THUtils.displayMsg("")
                end
                return cx, cy, cz, upX, upY, upZ, radius
            end
        end
    end
end
function THBetterWinch.overwrite_readSplitShapeIdFromStream(self, superFunc, streamId, ...)
    if streamReadBool(streamId) then
        local objectNodeId = 0
        if streamReadBool(streamId) then
            local object = NetworkUtil.readNodeObject(streamId)
            local componentIndex = streamReadUIntN(streamId, 16)
            local targetedMeshIndex, radius, upAxis = nil, nil, nil
            if streamReadBool(streamId) then
                targetedMeshIndex = streamReadUIntN(streamId, 16)
                radius = streamReadFloat32(streamId)
                upAxis = streamReadUIntN(streamId, 4)
            end
            if object ~= nil then
                THUtils.pcall(function()
                    local objectData = self:registerAttachableObject(object)
                    if objectData ~= nil then
                        local componentData = objectData:getComponentByIndex(componentIndex)
                        if componentData ~= nil then
                            objectData.lastTargetedComponentIndex = componentData.index
                            objectData.lastTargetedMeshNodeIndex = nil
                            if targetedMeshIndex ~= nil then
                                local targetedMeshData = objectData:getMeshNodeByIndex(targetedMeshIndex)
                                if targetedMeshData ~= nil then
                                    objectData.lastTargetedMeshNodeIndex = targetedMeshData.index
                                    if radius ~= nil and radius > 0 then
                                        targetedMeshData.radius = radius
                                    end
                                    if upAxis ~= nil and upAxis > 0 then
                                        targetedMeshData.upAxis = upAxis
                                    end
                                end
                            end
                            objectNodeId = componentData.node
                        end
                    end
                end)
            end
        end
        return objectNodeId, 0, 0
    end
    return superFunc(streamId, ...)
end
function THBetterWinch.overwrite_writeSplitShapeIdToStream(self, superFunc, streamId, objectNodeId, ...)
    local componentData = nil
    local targetedMeshData = nil
    THUtils.pcall(function()
        local objectData = self:getAttachableObjectByComponentNode(objectNodeId)
        if objectData ~= nil then
            local _ = nil
            componentData = objectData:getComponent(objectNodeId, false, true)
            _, targetedMeshData = objectData:getLastTargetData()
        end
    end)
    if streamWriteBool(streamId, componentData ~= nil) then
        if streamWriteBool(streamId, THUtils.getNodeExists(componentData.node)) then
            local objectData = componentData.parent
            NetworkUtil.writeNodeObject(streamId, objectData.target)
            streamWriteUIntN(streamId, componentData.index, 16)
            if streamWriteBool(streamId, targetedMeshData ~= nil) then
                streamWriteUIntN(streamId, targetedMeshData.index, 16)
                streamWriteFloat32(streamId, targetedMeshData.radius or 0)
                streamWriteUIntN(streamId, targetedMeshData.upAxis or 0, 4)
            end
            return componentData.node, 0, 0
        end
        return 0, 0, 0
    end
    return superFunc(streamId, objectNodeId, ...)
end
function THBetterWinch.overwrite_getTreeOffsetPosition(self, superFunc, nodeId, x, y, z, maxRadius, minLength, ...)
    local objectData = nil
    THUtils.pcall(function()
        if THUtils.getNodeExists(nodeId) then
            objectData = self:getAttachableObjectByNodeId(nodeId)
        end
    end)
    if objectData ~= nil then
        return self:getAttachableObjectOffsetPosition(nodeId, x, y, z, maxRadius, minLength)
    end
    return superFunc(nodeId, x, y, z, maxRadius, minLength, ...)
end
function THBetterWinch.overwrite_createSplitShapeTreeBelt(self, superFunc, beltData, objectNodeId, tx, ty, tz, sx, sy, sz, upX, upY, upZ, hookOffset, ignoreYDirection, spacing, ...)
    if objectNodeId ~= nil then
        THUtils.pcall(function()
            local objectData, isMeshNode = self:getAttachableObjectByNodeId(objectNodeId)
            if objectData ~= nil then
                local meshData = nil
                if isMeshNode then
                    meshData = objectData:getMeshNode(objectNodeId, false, true)
                else
                    local _ = nil
                    _, meshData = objectData:getLastTargetData()
                end
                if meshData ~= nil then
                    objectNodeId = meshData.node
                end
            end
        end)
    end
    return superFunc(beltData, objectNodeId, tx, ty, tz, sx, sy, sz, upX, upY, upZ, hookOffset, ignoreYDirection, spacing, ...)
end
THUtils.pcall(initScript)