-- FS25_Wet_Wheel_Tracks/scripts/WetWheelTracks.lua
-- Slippery activates only on REAL slip and/or slow work mode.
-- Road driving stays stable even in heavy rain.
-- GripIntensity (0.2..2.5) now has a CLEAR, strong effect.

WetWheelTracks = WetWheelTracks or {}

----------------------------------------------------------------
-- CONFIG (tweak these)
----------------------------------------------------------------

-- Vehicle type baseline: how low grip can go at wet=1.0 (before strength shaping)
WetWheelTracks.vehicleTypeConfig = {
    tractor   = { minGripWet = 0.60 },
    harvester = { minGripWet = 0.65 },
    truck     = { minGripWet = 0.55 },
    car       = { minGripWet = 0.70 },
    default   = { minGripWet = 0.60 },
}

-- Debug
WetWheelTracks.debugPrintEnabled = false
WetWheelTracks.debugIntervalMs  = 1000

-- Slip detection thresholds (more sensitive than before)
-- slipRatio = |wheelLinearSpeed - vehicleSpeed| / max(vehicleSpeed, 0.5m/s)
WetWheelTracks.SLIP_ON   = 0.08  -- start applying when above this
WetWheelTracks.SLIP_FULL = 0.25  -- full slip factor at/above this

-- Road/Speed filter (km/h): above OFF => no effect (stable road)
WetWheelTracks.SPEED_ON_KMH  = 8
WetWheelTracks.SPEED_OFF_KMH = 22

-- Slow work mode assist:
WetWheelTracks.WORK_MAX_KMH = 12

-- Minimum wetness to even consider slippery
WetWheelTracks.MIN_WETNESS = 0.05

-- Safety clamp for lowest possible grip (so it never becomes absurd)
WetWheelTracks.ABS_MIN_GRIP = 0.15

-- Internal
WetWheelTracks._initialized  = false
WetWheelTracks._debugTimerMs = 0

local function clamp(x, a, b)
    if x < a then return a end
    if x > b then return b end
    return x
end

local function lerp(a, b, t)
    return a + (b - a) * t
end

local function kmhToMs(kmh)
    return kmh * (1000 / 3600)
end

----------------------------------------------------------------
-- INIT
----------------------------------------------------------------
function WetWheelTracks:init()
    self._initialized = true
end

----------------------------------------------------------------
-- SETTINGS: effect strength from slider (0.2..2.5)
-- Returns 0.25..2.0 (strong and noticeable)
----------------------------------------------------------------
function WetWheelTracks:getEffectStrength()
    local intensity = 1.0
    if WetWheelTracks.settings and WetWheelTracks.settings.getGripIntensity then
        intensity = WetWheelTracks.settings:getGripIntensity()
    end

    -- Map slider [0.2..2.5] -> strength [0.25..2.0]
    local t = (clamp(intensity, 0.2, 2.5) - 0.2) / (2.5 - 0.2)  -- 0..1
    return lerp(0.25, 2.0, t)
end

----------------------------------------------------------------
-- VEHICLE TYPE (simple heuristic)
----------------------------------------------------------------
function WetWheelTracks:getVehicleTypeKey(vehicle)
    if vehicle == nil or vehicle.typeName == nil then
        return "default"
    end

    local t = string.lower(tostring(vehicle.typeName))
    if string.find(t, "tractor") then return "tractor" end
    if string.find(t, "harvester") or string.find(t, "combine") then return "harvester" end
    if string.find(t, "truck") then return "truck" end
    if string.find(t, "car") or string.find(t, "pickup") then return "car" end
    return "default"
end

----------------------------------------------------------------
-- WETNESS
----------------------------------------------------------------
function WetWheelTracks:getWetness()
    if g_currentMission == nil or g_currentMission.environment == nil then
        return 0
    end

    local env = g_currentMission.environment
    local weather = env.weather
    if weather ~= nil and weather.getGroundWetness ~= nil then
        return clamp(weather:getGroundWetness() or 0, 0, 1)
    end

    return 0
end

----------------------------------------------------------------
-- VEHICLE SPEED (m/s)
----------------------------------------------------------------
function WetWheelTracks:getVehicleSpeedMs(vehicle)
    if vehicle == nil then
        return 0
    end

    -- GIANTS: getLastSpeed() on käytännössä km/h
    if vehicle.getLastSpeed ~= nil then
        local vKmh = vehicle:getLastSpeed()
        if type(vKmh) == "number" then
            if vKmh < 0 then vKmh = 0 end
            return vKmh / 3.6
        end
    end

    -- Fallback: joissain tapauksissa lastSpeedReal voi olla jo m/s (tai km/h),
    -- joten tehdään varovainen muunnos jos vaikuttaa km/h:lta.
    if type(vehicle.lastSpeedReal) == "number" then
        local v = vehicle.lastSpeedReal
        if v < 0 then v = 0 end

        -- Heuristiikka: jos arvo näyttää km/h:lta (esim > 15), muunna
        -- (15 m/s = 54 km/h, joten tämä ei riko normaaleja työnopeuksia)
        if v > 15 then
            return v / 3.6
        end

        return v
    end

    return 0
end

----------------------------------------------------------------
-- WORK MODE (best-effort, no terrain needed)
----------------------------------------------------------------
function WetWheelTracks:isWorkActive(vehicle)
    if vehicle == nil then
        return false
    end

    if vehicle.getIsLowered ~= nil and vehicle:getIsLowered() then
        return true
    end
    if vehicle.getIsTurnedOn ~= nil and vehicle:getIsTurnedOn() then
        return true
    end
    if vehicle.getIsOperating ~= nil and vehicle:getIsOperating() then
        return true
    end

    -- Fallback: power consumer activity (best effort)
    if vehicle.spec_powerConsumer ~= nil then
        local spec = vehicle.spec_powerConsumer
        if spec.getIsPowered ~= nil and spec:getIsPowered() then
            return true
        end
        if spec.isActive ~= nil and spec.isActive then
            return true
        end
    end

    return false
end

----------------------------------------------------------------
-- GRIP CURVE (wetness -> targetGripScale, strength applied here)
-- baseMinGripWet is a "design baseline", strength makes it more/less effective.
----------------------------------------------------------------
function WetWheelTracks:computeWetTargetGrip(wet, vehicleTypeKey, strength)
    local cfg = self.vehicleTypeConfig[vehicleTypeKey] or self.vehicleTypeConfig.default
    local baseMin = clamp(cfg.minGripWet or 0.60, 0.30, 1.00)

    if wet <= 0 then
        return 1.0
    end

    local w = clamp(wet, 0, 1)
    local wPow = w ^ 1.5

    -- Baseline (strength=1)
    local baseGrip = 1.0 - (1.0 - baseMin) * wPow  -- [baseMin..1]
    baseGrip = clamp(baseGrip, baseMin, 1.0)

    -- Apply strength to the *reduction* amount:
    -- reduction = (1 - baseGrip) * strength
    local reduction = (1.0 - baseGrip) * clamp(strength, 0.25, 2.0)
    local grip = 1.0 - reduction

    return clamp(grip, WetWheelTracks.ABS_MIN_GRIP, 1.0)
end

----------------------------------------------------------------
-- WHEEL EXTRA (width / duals)
----------------------------------------------------------------
function WetWheelTracks:computeWheelExtraFactor(wp)
    if wp == nil or type(wp) ~= "table" then
        return 1.0
    end

    local width =
          wp.width
       or wp.contactWidth
       or wp.tireWidth
       or wp.wheelWidth
       or 0.5

    local factor = 1.0
    if width <= 0.30 then
        factor = 0.85
    elseif width <= 0.40 then
        factor = 0.95
    elseif width <= 0.60 then
        factor = 1.00
    elseif width <= 0.80 then
        factor = 1.08
    else
        factor = 1.12
    end

    local isDual =
           (wp.additionalWheelIndex ~= nil)
        or (wp.twinIndex ~= nil)
        or (wp.isTwin ~= nil and wp.isTwin == true)
        or false

    if isDual then
        factor = factor * 1.08
    end

    return clamp(factor, 0.80, 1.30)
end

----------------------------------------------------------------
-- WHEEL ANGULAR SPEED (rad/s) best-effort
----------------------------------------------------------------
function WetWheelTracks:getWheelAngularSpeed(wp)
    if wp == nil or type(wp) ~= "table" then
        return 0
    end

    if type(wp.angularVelocity) == "number" then
        return wp.angularVelocity
    end
    if type(wp.rotSpeed) == "number" then
        return wp.rotSpeed
    end

    local wheelShape = wp.wheelShape or wp.wheelShapeId
    if wheelShape ~= nil and getWheelShapeRotationSpeed ~= nil then
        local ok, v = pcall(getWheelShapeRotationSpeed, wheelShape)
        if ok and type(v) == "number" then
            return v
        end
    end

    if wheelShape ~= nil and getWheelShapeAxleSpeed ~= nil then
        local ok, v = pcall(getWheelShapeAxleSpeed, wheelShape)
        if ok and type(v) == "number" then
            return v
        end
    end

    return 0
end

function WetWheelTracks:getWheelRadius(wp)
    if wp == nil or type(wp) ~= "table" then
        return 0.6
    end
    if type(wp.radius) == "number" then
        return wp.radius
    end
    if type(wp.wheelRadius) == "number" then
        return wp.wheelRadius
    end
    return 0.6
end

----------------------------------------------------------------
-- ACTIVATION FACTORS
----------------------------------------------------------------
function WetWheelTracks:getSpeedFactor(speedMs)
    local onMs  = kmhToMs(self.SPEED_ON_KMH)
    local offMs = kmhToMs(self.SPEED_OFF_KMH)
    if offMs <= onMs then
        offMs = onMs + 0.1
    end

    if speedMs >= offMs then
        return 0.0
    end
    if speedMs <= onMs then
        return 1.0
    end

    local t = (speedMs - onMs) / (offMs - onMs)
    return clamp(1.0 - t, 0.0, 1.0)
end

function WetWheelTracks:getSlipFactor(slipRatio)
    local on  = self.SLIP_ON
    local full = self.SLIP_FULL
    if full <= on then
        full = on + 0.01
    end

    if slipRatio <= on then
        return 0.0
    end
    if slipRatio >= full then
        return 1.0
    end

    return clamp((slipRatio - on) / (full - on), 0.0, 1.0)
end

function WetWheelTracks:getWorkFactor(vehicle, speedMs)
    if self:isWorkActive(vehicle) then
        return 1.0
    end

    local workMaxMs = kmhToMs(self.WORK_MAX_KMH)
    if speedMs <= workMaxMs then
        return 1.0
    end

    -- Not working and faster than WORK_MAX: allow effect only if slipping, but reduced
    return 0.35
end

----------------------------------------------------------------
-- APPLY FRICTION (blend 1.0 -> targetScale by activation)
----------------------------------------------------------------
function WetWheelTracks:applyWheelFrictionBlend(wp, targetScale, blendAlpha)
    if wp == nil or type(wp) ~= "table" then
        return
    end

    if wp._wetWheelTracksBase == nil then
        wp._wetWheelTracksBase = {
            maxLatFriction  = wp.maxLatFriction,
            maxLongFriction = wp.maxLongFriction,
            frictionScale   = wp.frictionScale
        }
    end

    local base = wp._wetWheelTracksBase
    if base == nil then
        return
    end

    local finalScale = lerp(1.0, targetScale, clamp(blendAlpha, 0.0, 1.0))

    if type(base.maxLatFriction) == "number" and type(wp.maxLatFriction) == "number" then
        wp.maxLatFriction = base.maxLatFriction * finalScale
    end
    if type(base.maxLongFriction) == "number" and type(wp.maxLongFriction) == "number" then
        wp.maxLongFriction = base.maxLongFriction * finalScale
    end
    if type(base.frictionScale) == "number" and type(wp.frictionScale) == "number" then
        wp.frictionScale = base.frictionScale * finalScale
    end
end

----------------------------------------------------------------
-- WheelPhysics HOOKS
----------------------------------------------------------------
function WetWheelTracks:onWheelPhysicsFinalize(wp)
    -- ensure base captured
    self:applyWheelFrictionBlend(wp, 1.0, 0.0)
end

function WetWheelTracks:onWheelPhysicsServerUpdate(wp, dt)
    if g_currentMission == nil then return end

    local vehicle = wp and (wp.vehicle or wp.parent) or nil
    local speedMs = self:getVehicleSpeedMs(vehicle)
    local wet = self:getWetness()

    -- Dry => no effect
    if wet < self.MIN_WETNESS then
        self:applyWheelFrictionBlend(wp, 1.0, 0.0)
        return
    end

    local strength = self:getEffectStrength()

    -- Slip estimation
    local ang = self:getWheelAngularSpeed(wp)
    local radius = self:getWheelRadius(wp)
    local wheelLin = math.abs(ang) * math.max(radius, 0.05)

    local denom = math.max(speedMs, 0.5)
    local slipRatio = math.abs(wheelLin - speedMs) / denom

    local slipFactor  = self:getSlipFactor(slipRatio)
    local speedFactor = self:getSpeedFactor(speedMs)
    local workFactor  = self:getWorkFactor(vehicle, speedMs)

    -- Activation: only when actually slipping, and never at road speed.
    -- Strength also scales activation so slider feels strong.
    local activation = slipFactor * speedFactor * workFactor
    activation = clamp(activation * clamp(strength, 0.25, 2.0), 0.0, 1.0)

    -- Target grip scale (strength shapes how low we can go)
    local vTypeKey = self:getVehicleTypeKey(vehicle)
    local wetGripScale = self:computeWetTargetGrip(wet, vTypeKey, strength)

    local wheelExtra = self:computeWheelExtraFactor(wp)
    local targetScale = clamp(wetGripScale * wheelExtra, WetWheelTracks.ABS_MIN_GRIP, 1.0)

    self:applyWheelFrictionBlend(wp, targetScale, activation)

    -- Debug 1x/s
    if self.debugPrintEnabled then
        self._debugTimerMs = (self._debugTimerMs or 0) + (dt or 0)
        if self._debugTimerMs >= (self.debugIntervalMs or 1000) then
            self._debugTimerMs = 0
            local kmh = speedMs * 3.6
            print(string.format(
                "[WetWheelTracks] wet=%.2f strength=%.2f speed=%.1fkm/h wheel=%.2fm/s slip=%.2f (sf=%.2f spf=%.2f wf=%.2f) act=%.2f target=%.2f",
                wet, strength, kmh, wheelLin, slipRatio, slipFactor, speedFactor, workFactor, activation, targetScale
            ))
        end
    end
end

----------------------------------------------------------------
-- INSTALL HOOKS
----------------------------------------------------------------
if WheelPhysics ~= nil then
    WheelPhysics.finalize = Utils.appendedFunction(WheelPhysics.finalize, function(self, ...)
        WetWheelTracks:onWheelPhysicsFinalize(self)
    end)

    WheelPhysics.serverUpdate = Utils.appendedFunction(WheelPhysics.serverUpdate, function(self, dt, ...)
        WetWheelTracks:onWheelPhysicsServerUpdate(self, dt)
    end)

    print("[WetWheelTracks] WheelPhysics hooks installed (finalize + serverUpdate).")
else
    print("[WetWheelTracks] WARNING: WheelPhysics not found!")
end
