-- FS25_WetWheelTracks/scripts/WetWheelTracksSettings.lua

WetWheelTracksSettings = {}
WetWheelTracksSettings.SETTINGS_FILE_NAME   = "fs25_wet_wheel_tracks"
WetWheelTracksSettings.I18N_PREFIX          = "wwt_"
WetWheelTracksSettings.SECTION_TITLE_ID     = WetWheelTracksSettings.I18N_PREFIX .. "settings_section_title"
WetWheelTracksSettings.GRIP_INTENSITY_KEY   = "gripIntensity"

WetWheelTracksSettings.VALUE_DEFAULT        = 1.0
WetWheelTracksSettings.VALUE_MIN            = 0.2
WetWheelTracksSettings.VALUE_MAX            = 2.5
WetWheelTracksSettings.VALUE_STEP           = 0.05

WetWheelTracksSettings.CONTROL = {
    name     = WetWheelTracksSettings.GRIP_INTENSITY_KEY,
    min      = WetWheelTracksSettings.VALUE_MIN,
    max      = WetWheelTracksSettings.VALUE_MAX,
    step     = WetWheelTracksSettings.VALUE_STEP,
    autoBind = true,
}

WetWheelTracksSettings.settingsLoadAttempted = false
WetWheelTracksSettings.isUiAdded             = false

local WetWheelTracksSettings_mt = Class(WetWheelTracksSettings)

function WetWheelTracksSettings.new()
    local self = setmetatable({}, WetWheelTracksSettings_mt)

    self.controls        = {} -- UIHelper käyttää tätä
    self.settings        = {}
    self.settings[WetWheelTracksSettings.GRIP_INTENSITY_KEY] = WetWheelTracksSettings.VALUE_DEFAULT
    self.settingsFromXml = {}

    return self
end

function WetWheelTracksSettings:getGripIntensity()
    return self.settings[WetWheelTracksSettings.GRIP_INTENSITY_KEY] or WetWheelTracksSettings.VALUE_DEFAULT
end

----------------------------------------------------------------
-- UI lisäys Game Settings -sivulle
----------------------------------------------------------------
function WetWheelTracksSettings:addSettingsUiToGame()
    if WetWheelTracksSettings.isUiAdded then
        return
    end

    local settingsPage = g_gui.screenControllers[InGameMenu].pageSettings

    UIHelper.createControlsDynamically(
        settingsPage,
        WetWheelTracksSettings.SECTION_TITLE_ID,
        self,
        { WetWheelTracksSettings.CONTROL },
        WetWheelTracksSettings.I18N_PREFIX
    )

    UIHelper.setupAutoBindControls(self, self.settings)

    WetWheelTracksSettings.isUiAdded = true

    print("[WetWheelTracks] Settings UI added to Game Settings.")
end

----------------------------------------------------------------
-- XML-lataus
----------------------------------------------------------------
function WetWheelTracksSettings:loadSettingsFromXml()
    if WetWheelTracksSettings.settingsLoadAttempted then
        return
    end

    local xmlFilePath = WetWheelTracksSettings.getXmlFilePath()

    if xmlFilePath == nil then
        WetWheelTracksSettings.settingsLoadAttempted = true
        self.settingsFromXml = self:getSettingsCopy()
        return
    end

    if not fileExists(xmlFilePath) then
        WetWheelTracksSettings.settingsLoadAttempted = true
        self.settingsFromXml = self:getSettingsCopy()
        return
    end

    local xmlFileId = loadXMLFile("WetWheelTracksSettings", xmlFilePath)
    if xmlFileId == 0 then
        WetWheelTracksSettings.settingsLoadAttempted = true
        Logging.warning("WetWheelTracksSettings: Failed reading from XML file")
        self.settingsFromXml = self:getSettingsCopy()
        return
    end

    local val = getXMLFloat(xmlFileId, "WetWheelTracks.gripIntensity")
                or self.settings[WetWheelTracksSettings.GRIP_INTENSITY_KEY]

    val = math.min(math.max(val, WetWheelTracksSettings.VALUE_MIN), WetWheelTracksSettings.VALUE_MAX)
    self.settings[WetWheelTracksSettings.GRIP_INTENSITY_KEY] = val

    self.settingsFromXml = self:getSettingsCopy()
    WetWheelTracksSettings.settingsLoadAttempted = true

    print(string.format("[WetWheelTracks] Loaded gripIntensity=%.2f from XML", val))
end

----------------------------------------------------------------
-- XML-tallennus vain jos muuttunut
----------------------------------------------------------------
function WetWheelTracksSettings:maybeSaveSettingsToXml()
    if self.settings[WetWheelTracksSettings.GRIP_INTENSITY_KEY] ~=
       self.settingsFromXml[WetWheelTracksSettings.GRIP_INTENSITY_KEY] then
        self:saveSettingsToXml()
    end
end

function WetWheelTracksSettings:saveSettingsToXml()
    local xmlFilePath = WetWheelTracksSettings.getXmlFilePath()
    if xmlFilePath == nil then
        return
    end

    local xmlFileId = createXMLFile("WetWheelTracksSettings", xmlFilePath, "WetWheelTracks")

    setXMLFloat(
        xmlFileId,
        "WetWheelTracks.gripIntensity",
        self.settings[WetWheelTracksSettings.GRIP_INTENSITY_KEY] or WetWheelTracksSettings.VALUE_DEFAULT
    )

    saveXMLFile(xmlFileId)

    self.settingsFromXml = self:getSettingsCopy()

    print(string.format("[WetWheelTracks] Saved gripIntensity=%.2f to XML %s",
        self.settings[WetWheelTracksSettings.GRIP_INTENSITY_KEY],
        xmlFilePath
    ))
end

function WetWheelTracksSettings:getSettingsCopy()
    local s = {}
    s[WetWheelTracksSettings.GRIP_INTENSITY_KEY] = self.settings[WetWheelTracksSettings.GRIP_INTENSITY_KEY]
    return s
end

function WetWheelTracksSettings.getXmlFilePath()
    if g_currentMission ~= nil and g_currentMission.missionInfo ~= nil then
        local savegameDirectory = g_currentMission.missionInfo.savegameDirectory
        if savegameDirectory ~= nil then
            return string.format("%s/%s.xml", savegameDirectory, WetWheelTracksSettings.SETTINGS_FILE_NAME)
        end
    end
    return nil
end
