-- RMC_SelectedToolControlGroup.lua
-- Control Group Selection Indicator (RMC)
-- Release build:
--   DEBUG = false          -> no debug spam
--   PRINT_STARTUP = false  -> no startup/install log lines

RMC_SelectedToolControlGroup = {}

--========================================================
-- Release toggles (true/false)
--========================================================
RMC_SelectedToolControlGroup.DEBUG = false
RMC_SelectedToolControlGroup.PRINT_STARTUP = false

RMC_SelectedToolControlGroup._installed = false
RMC_SelectedToolControlGroup._lastDebugPrint = 0

local function logStartup(fmt, ...)
    if RMC_SelectedToolControlGroup.PRINT_STARTUP or RMC_SelectedToolControlGroup.DEBUG then
        print(string.format("[RMC_SelectedToolControlGroup] " .. fmt, ...))
    end
end

local function dbg(fmt, ...)
    if RMC_SelectedToolControlGroup.DEBUG then
        print(string.format("[RMC_SelectedToolControlGroup] " .. fmt, ...))
    end
end

--========================================================
-- Store name helpers
--========================================================

local function getStoreNameFromObject(obj)
    if obj == nil then
        return nil, "obj=nil"
    end

    local xmlFilename = obj.configFileName or obj.xmlFilename

    if xmlFilename ~= nil and g_storeManager ~= nil and g_storeManager.getItemByXMLFilename ~= nil then
        local item = g_storeManager:getItemByXMLFilename(xmlFilename)
        if item ~= nil and item.name ~= nil and item.name ~= "" then
            return item.name, "storeManager"
        end
    end

    if obj.getStoreItem ~= nil then
        local item = obj:getStoreItem()
        if item ~= nil and item.name ~= nil and item.name ~= "" then
            return item.name, "getStoreItem"
        end
    end

    return nil, string.format("noName xml=%s", tostring(xmlFilename))
end

--========================================================
-- Attachment traversal / duplicate numbering
--========================================================

local function collectAttachedVehiclesRecursive(vehicle, out, depth)
    if vehicle == nil or depth > 10 then
        return
    end

    out[#out + 1] = vehicle

    if vehicle.getAttachedImplements ~= nil then
        for _, imp in pairs(vehicle:getAttachedImplements()) do
            local obj = imp.object
            if obj ~= nil then
                collectAttachedVehiclesRecursive(obj, out, depth + 1)
            end
        end
    end
end

local function getDuplicateIndexForSelected(rootVehicle, selectedObj, selectedName)
    if rootVehicle == nil or selectedObj == nil or selectedName == nil or selectedName == "" then
        return nil
    end
    if selectedObj == rootVehicle then
        return nil
    end

    local all = {}
    collectAttachedVehiclesRecursive(rootVehicle, all, 0)

    local matchCount = 0
    local myIndex = nil

    for _, v in ipairs(all) do
        if v ~= rootVehicle then
            local name = getStoreNameFromObject(v)
            if name == selectedName then
                matchCount = matchCount + 1
                if v == selectedObj then
                    myIndex = matchCount
                end
            end
        end
    end

    if matchCount > 1 and myIndex ~= nil then
        return myIndex
    end

    return nil
end

--========================================================
-- Selection resolution (nested-safe)
--========================================================

local function findSelectedImplementRecursive(vehicle, depth)
    if vehicle == nil or depth > 6 then
        return nil
    end

    if vehicle.getSelectedImplement ~= nil then
        local imp = vehicle:getSelectedImplement()
        if imp ~= nil then
            return imp.object or imp
        end
    end

    if vehicle.getAttachedImplements ~= nil then
        for _, imp in pairs(vehicle:getAttachedImplements()) do
            local obj = imp.object
            if obj ~= nil then
                local found = findSelectedImplementRecursive(obj, depth + 1)
                if found ~= nil then
                    return found
                end
            end
        end
    end

    return nil
end

-- Returns: name, how, xml, selectedObj
local function getCurrentlySelectedName(vehicle)
    if vehicle == nil then
        return nil, "vehicle=nil", nil, nil
    end

    local root = vehicle.rootVehicle or vehicle

    -- 1) Best: chain-wide selection
    if root.getSelectedVehicle ~= nil then
        local sel = root:getSelectedVehicle()
        if sel == nil then
            sel = root -- treat nil as root selected
        end

        local name, src = getStoreNameFromObject(sel)
        local xml = sel.configFileName or sel.xmlFilename
        return name, "selectedVehicle/" .. tostring(src), xml, sel
    end

    -- 2) Fallback: root selected implement
    if root.getSelectedImplement ~= nil then
        local imp = root:getSelectedImplement()
        if imp ~= nil then
            local obj = imp.object or imp
            local name, src = getStoreNameFromObject(obj)
            local xml = obj and (obj.configFileName or obj.xmlFilename) or nil
            return name, "selectedImplement/" .. tostring(src), xml, obj
        end
    end

    -- 3) Safety net: recurse attachments
    local deepObj = findSelectedImplementRecursive(root, 0)
    if deepObj ~= nil then
        local name, src = getStoreNameFromObject(deepObj)
        local xml = deepObj.configFileName or deepObj.xmlFilename
        return name, "recursive/" .. tostring(src), xml, deepObj
    end

    -- 4) Final: root name
    local rootName, rootSrc = getStoreNameFromObject(root)
    local rootXml = root.configFileName or root.xmlFilename
    return rootName, "rootFallback/" .. tostring(rootSrc), rootXml, root
end

local function buildDisplayName(vehicle)
    if vehicle == nil then
        return nil, nil, nil, nil, nil
    end

    local root = vehicle.rootVehicle or vehicle
    local selectedName, how, xml, selectedObj = getCurrentlySelectedName(vehicle)

    local displayName = selectedName
    if displayName ~= nil and displayName ~= "" and selectedObj ~= nil then
        local idx = getDuplicateIndexForSelected(root, selectedObj, selectedName)
        if idx ~= nil then
            displayName = string.format("%s %d", selectedName, idx)
        end
    end

    return displayName, how, xml, selectedObj, root
end

--========================================================
-- Text truncation (earlier by a small pad)
--========================================================

local function limitHeaderText(self, text)
    if text == nil then
        return ""
    end

    local bg = self.comboBg
    if bg == nil then
        return text
    end

    local offsetX = self.comboTextOffsetX or 0
    local extraPad = (self.scalePixelToScreenWidth ~= nil) and self:scalePixelToScreenWidth(18) or (g_pixelSizeX * 18)

    local maxWidth = bg.width - 2 * offsetX - extraPad

    -- Clamp so we never truncate down to empty on weird UI scales
    local minWidth = (self.scalePixelToScreenWidth ~= nil) and self:scalePixelToScreenWidth(60) or (g_pixelSizeX * 60)
    if maxWidth < minWidth then
        maxWidth = minWidth
    end

    return Utils.limitTextToWidth(text, self.textSize, maxWidth, false, "...")
end

--========================================================
-- Selected icon highlight overlay
--========================================================

local function ensureHighlightOverlay(self)
    if self.rmcSelectedBgOverlay == nil then
        self.rmcSelectedBgOverlay = g_overlayManager:createOverlay(g_plainColorSliceId, 0, 0, 0, 0)
    end
end

local function renderSelectedBg(self, x, y, w, h)
    ensureHighlightOverlay(self)
    local c = HUD.COLOR.ACTIVE or { 0.2, 0.8, 1.0, 1.0 }
    self.rmcSelectedBgOverlay:renderCustom(x, y, w, h, c[1], c[2], c[3], 0.22)
end

--========================================================
-- Install hooks (draw + drawVehicleSchema)
--========================================================

function RMC_SelectedToolControlGroup.install()
    if RMC_SelectedToolControlGroup._installed then
        return
    end

    if InputHelpDisplay == nil or InputHelpDisplay.drawVehicleSchema == nil or InputHelpDisplay.draw == nil then
        return
    end

    RMC_SelectedToolControlGroup._installed = true

    -- Wrap draw() to inject bottom line when a TOOL is selected (not root)
    local superDraw = InputHelpDisplay.draw
    InputHelpDisplay.draw = function(self, offsetX, offsetY)
        if self ~= nil and self.vehicle ~= nil and self.getVisible ~= nil and self:getVisible() then
            local displayName, how, xml, selectedObj, root = buildDisplayName(self.vehicle)

            if displayName ~= nil and displayName ~= "" and selectedObj ~= nil and root ~= nil and selectedObj ~= root then
                self.extraHelpTexts = self.extraHelpTexts or {}
                self.extraHelpTexts["rmc_selected_tool_name"] = displayName
            end

            if RMC_SelectedToolControlGroup.DEBUG then
                local now = g_time or 0
                if now - RMC_SelectedToolControlGroup._lastDebugPrint > 1000 then
                    RMC_SelectedToolControlGroup._lastDebugPrint = now
                    dbg("extraLine='%s' selectedIsRoot=%s how=%s xml=%s",
                        tostring(displayName),
                        tostring(selectedObj == root),
                        tostring(how),
                        tostring(xml))
                end
            end
        end

        return superDraw(self, offsetX, offsetY)
    end

    -- Override drawVehicleSchema for header text + schema highlight
    InputHelpDisplay.drawVehicleSchema = function(self, posX, posY, isShortVersion)
        local vehicle = self.vehicle
        if vehicle == nil or vehicle.schemaOverlay == nil then
            return posY, nil
        end

        self:getVehicleSchemaOverlays(vehicle.rootVehicle)
        local minX, maxX = self:getSchemaDelimiters()

        local topY

        if isShortVersion then
            topY = posY - self.lineBg.height
            local w = maxX - minX + 2 * self.schemaOffsetX - 2 * self.lineBgLeft.width
            self.lineBgScale:setDimension(w, nil)
            self.lineBgLeft:setPosition(posX, topY)
            self.lineBgScale:setPosition(self.lineBgLeft.x + self.lineBgLeft.width, topY)
            self.lineBgRight:setPosition(self.lineBgScale.x + self.lineBgScale.width, topY)
            self.lineBgLeft:render()
            self.lineBgScale:render()
            self.lineBgRight:render()
        else
            topY = posY - self.comboBg.height

            local baseText = utf8ToUpper(g_i18n:getText("ui_controlsControlGroup"))
            local displayName = buildDisplayName(vehicle)

            local headerText
            if displayName ~= nil and displayName ~= "" then
                headerText = string.format("%s (%s)", baseText, utf8ToUpper(displayName))
            else
                headerText = baseText
            end

            self.controlGroupText = limitHeaderText(self, headerText)

            self.comboBg:renderCustom(posX, topY)
            setTextBold(true)
            setTextAlignment(RenderText.ALIGN_LEFT)
            setTextColor(1, 1, 1, 1)
            renderText(posX + self.comboTextOffsetX, topY + self.comboTextOffsetY, self.textSize, self.controlGroupText)
            self.separatorHorizontal:renderCustom(posX + self.comboSeparatorOffsetX, topY + self.comboSeparatorOffsetY)
        end

        local schemaWidth = maxX - minX
        local scale = (self.maxSchemaWidth >= schemaWidth) and 1 or (self.maxSchemaWidth / schemaWidth)
        local baseX = posX - minX + self.schemaOffsetX
        local baseY = topY + self.schemaOffsetY

        -- Highlight padding (10% smaller box)
        local padScale = 0.9
        local padX = (self.scalePixelToScreenWidth ~= nil) and self:scalePixelToScreenWidth(2 * padScale) or (g_pixelSizeX * 2 * padScale)
        local padY = (self.scalePixelToScreenHeight ~= nil) and self:scalePixelToScreenHeight(2 * padScale) or (g_pixelSizeY * 2 * padScale)

        for _, entry in ipairs(self.schemaOverlayEntryCache) do
            if entry.isUsed then
                local overlay = entry.overlay
                local ow, oh = overlay.width, overlay.height

                overlay:setInvertX(entry.invertX)

                local ox = baseX + entry.x
                local oy = baseY + entry.y

                overlay:setPosition(ox, oy)
                overlay:setRotation(entry.rotation, 0, 0)
                overlay:setDimension(ow * scale, oh * scale)


                -- Selection/Power coloring rules:
                -- Power ON keeps the original (classic FS) green.
                -- Selection uses HUD active color and *overrides* power-on color when selected.
                local powerOnColor = { 0.12, 1.0, 0.12, 1.0 }
                local selectedColor = HUD.COLOR.ACTIVE or { 0.2, 0.8, 1.0, 1.0 }

                local baseColor = entry.turnedOn and powerOnColor or HUD.COLOR.DEFAULT
                local color = entry.selected and selectedColor or baseColor

                -- Visibility: selected = full, power-on = full, off = dim
                local alpha = entry.selected and 1 or (entry.turnedOn and 1 or 0.5)
                overlay:setColor(color[1], color[2], color[3], alpha)
                overlay:render()

                if entry.additionalText ~= nil then
                    local tx = baseX + entry.x + ow * scale * 0.5
                    local ty = baseY + entry.y + oh * scale * 0.85
                    setTextBold(false)
                    setTextColor(1, 1, 1, 1)
                    setTextAlignment(RenderText.ALIGN_CENTER)
                    renderText(tx, ty, getCorrectTextSize(0.008), entry.additionalText)
                    setTextAlignment(RenderText.ALIGN_LEFT)
                    setTextColor(1, 1, 1, 1)
                end

                overlay:setDimension(ow, oh)
            end

            entry.isUsed = false
        end

        self.schemaOverlayEntryCacheIndex = 0
        return baseY - self.lineOffsetY, posY
    end

    logStartup("Installed hooks successfully (DEBUG=%s, PRINT_STARTUP=%s)",
        tostring(RMC_SelectedToolControlGroup.DEBUG),
        tostring(RMC_SelectedToolControlGroup.PRINT_STARTUP))
end

addModEventListener({
    loadMap = function()
        logStartup("loadMap() - script loaded")
    end,

    update = function()
        if not RMC_SelectedToolControlGroup._installed then
            RMC_SelectedToolControlGroup.install()
        end
    end
})
